/**
 *      \file   ltc2945.cpp
 *
 *      \brief  Class for LTC2945 power monitor chip.
 *
 *	\author Michael Williamson <michael.williamson@criticallink.com>
 *
 *     o  0
 *     | /       Copyright (c) 2014
 *    (CL)---o   Critical Link, LLC
 *      \
 *       O
 */
#include "ltc2945.h"
#include "i2c-dev.h"

#include <stdio.h>
#include <stdlib.h>

#define REG_DELTA_SENSE_MSB 0x14
#define REG_DELTA_SENSE_LSB 0x15

using namespace MityDSP;

tcLTC2945::tcLTC2945(int anBus, int anAddr, float Rsens, float Vin)
: tcI2CDevice(anBus, anAddr)
, mnVin(Vin)
, mnRsens(Rsens)
{
}

int tcLTC2945::GetDeltaSense(float& volts)
{
	int temp, val;
	temp = i2c_smbus_read_byte_data(mnFd, REG_DELTA_SENSE_MSB);
	if (temp < 0)
	{
		perror(__func__);
		return temp;
	}
	val |= temp << 8;
	val &= 0x00FF00;

	temp = i2c_smbus_read_byte_data(mnFd, REG_DELTA_SENSE_LSB);
	if (temp < 0)
	{
		perror("%s:");
		return temp;
	}
	val |= (temp & 0x00FF);
	val >>= 4;
	volts = val * 0.000025;
	return 0;
}

int tcLTC2945::Initialize(void)
{
    int  ret;
    char config;
    return Open();
}

int tcLTC2945::GetPowerW(float& power)
{
	int rv;
	float volts;
	rv = GetDeltaSense(volts);
	if (rv)
		return rv;

	float current = volts/mnRsens;
	power = current*mnVin;
	return 0;
}

#ifdef LTC2945_APP

#include <stdlib.h>
#include <unistd.h>
#include <string.h>

using namespace MityDSP;

int main(int argc, char* argv[])
{
	tcLTC2945* lpTmp;
	int addr = 0x6F; // 110 1111
	int bus = 1;
	int opt;
	opterr = 0;
	int rv;
	float Rsens = 0.01;
	float Vin = 5.00;

	while ((opt = getopt(argc, argv, "a:b:r:v:h")) != -1)
	{
		switch (opt)
		{
		case 'a':
			sscanf(optarg,"%x",&addr);
			break;

		case 'b':
			bus = strtoul(optarg,NULL,10);
			break;

		case 'v':
			sscanf(optarg,"%f", &Vin);
			break;

		case 'r':
			sscanf(optarg,"%f", &Rsens);
			break;

		case 'h': /* '?' */
		default: /* '?' */
			fprintf(stderr, "Usage: %s [-b <busnum>] [-a <addr>] [-v <Vin>] [-r <Rsens>]\n", argv[0]);
			fprintf(stderr, "Defaults: bus = %d, addr=0x%x\n", bus, addr);
			fprintf(stderr, "Defaults: Vin = %f, Rsens=0x%f\n", Vin, Rsens);
			return -1;
		}
	}

	float power;
	float volts;

	lpTmp = new tcLTC2945(bus, addr, Rsens, Vin);
	lpTmp->Initialize();

	rv = lpTmp->GetDeltaSense(volts);
	rv = lpTmp->GetPowerW(power);
	if (rv)
		fprintf(stderr, "unable to read power [%d, 0x%x]\n", rv, rv);
	else
		printf("Read %.5f volts on Vsens, computed power is %.3f W\n", volts, power);

	delete lpTmp;

	return 0;

}

#endif
