#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/ioport.h>
#include <linux/io.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/semaphore.h>
#include <linux/interrupt.h>
#include <linux/spinlock_types.h>
//
#include <linux/irq.h>
#include <linux/gpio.h>

#define UINPUT_BASE 0xff2000a0  //fpga_irq address
#define UINPUT_SIZE 0x10

#define UINPUT_INT_NUM 352      

static int gpio_number = UINPUT_INT_NUM;     

void *fpga_uinput_mem;

static DEFINE_SEMAPHORE(interrupt_mutex);
static DECLARE_WAIT_QUEUE_HEAD(interrupt_wq);

static int interrupt_flag = 0;
static DEFINE_SPINLOCK(interrupt_flag_lock);
static uint8_t input_state;

static irqreturn_t fpga_uinput_interrupt(int irq, void *dev_id)
{
   int irq_number;
   irq_number = gpio_to_irq(gpio_number);
   if (irq != irq_number)
   {
      printk(KERN_ALERT "irq != UINPUT_INT_NUM: Passed %s %d \n", __FUNCTION__, __LINE__);
      return IRQ_NONE;
   }

   spin_lock(&interrupt_flag_lock);
   interrupt_flag = 1;
   input_state = ioread8(fpga_uinput_mem);
   spin_unlock(&interrupt_flag_lock);

   wake_up_interruptible(&interrupt_wq);

   return IRQ_HANDLED;
}

static struct device_driver fpga_uinput_driver = {
   .name = "fpga_uinput0",
   .bus = &platform_bus_type,
};

static ssize_t fpga_uinput_show(struct device_driver *drv, char *buf)
{
   int ret;

   if (down_trylock(&interrupt_mutex))
      return -EAGAIN;

   if (wait_event_interruptible(interrupt_wq, interrupt_flag != 0))
   {
      ret = -ERESTART;
      goto release_and_exit;
   }

   spin_lock(&interrupt_flag_lock);
   interrupt_flag = 0;
   spin_unlock(&interrupt_flag_lock);

   buf[0] = input_state;
   ret = 1;

   release_and_exit:
   up(&interrupt_mutex);
   return ret;
}

static ssize_t fpga_uinput_store(struct device_driver *drv, const char *buf, size_t count)
{
   return -EROFS;
}

static DRIVER_ATTR(fpga_uinput, S_IRUSR, fpga_uinput_show, fpga_uinput_store);

static int __init fpga_uinput_init(void)
{
   int ret;
   int irq_number;
   struct resource *res;

   ret = driver_register(&fpga_uinput_driver);
   if (ret < 0)
      goto fail_driver_register;

   ret = driver_create_file(&fpga_uinput_driver, &driver_attr_fpga_uinput);
   if (ret < 0)
      goto fail_create_file;

   res = request_mem_region(UINPUT_BASE, UINPUT_SIZE, "fpga_uinput0");
   if (res == NULL)
   {
      ret = -EBUSY;
      goto fail_request_mem;
   }

   fpga_uinput_mem = ioremap(UINPUT_BASE, UINPUT_SIZE);
   if (fpga_uinput_mem == NULL)
   {
      ret = -EFAULT;
      goto fail_ioremap;
   }

   irq_number = gpio_to_irq(gpio_number);
   ret = request_irq(irq_number, fpga_uinput_interrupt, 0, "fpga_uinput0", NULL);
   if (ret < 0)
      goto fail_request_irq;

   return 0;

 fail_request_irq:
   iounmap(fpga_uinput_mem);
   printk(KERN_ALERT "DEBUG: fail_request_irq: Passed %s %d \n", __FUNCTION__, __LINE__);
 fail_ioremap:
   release_mem_region(UINPUT_BASE, UINPUT_SIZE);
   printk(KERN_ALERT "DEBUG: fail_ioremap: Passed %s %d \n", __FUNCTION__, __LINE__);
 fail_request_mem:
   driver_remove_file(&fpga_uinput_driver, &driver_attr_fpga_uinput);
   printk(KERN_ALERT "DEBUG: fail_request_mem: Passed %s %d \n", __FUNCTION__, __LINE__);
 fail_create_file:
   driver_unregister(&fpga_uinput_driver);
   printk(KERN_ALERT "DEBUG: fail_create_file: Passed %s %d \n", __FUNCTION__, __LINE__);
 fail_driver_register:
   printk(KERN_ALERT "DEBUG: fail_driver_register: Passed %s %d \n", __FUNCTION__, __LINE__);
   return ret;
}

static void __exit fpga_uinput_exit(void)
{
   free_irq(gpio_to_irq(gpio_number), NULL);
   iounmap(fpga_uinput_mem);
   release_mem_region(UINPUT_BASE, UINPUT_SIZE);
   driver_remove_file(&fpga_uinput_driver, &driver_attr_fpga_uinput);
   driver_unregister(&fpga_uinput_driver);
}

MODULE_LICENSE("Dual BSD/GPL");

module_init(fpga_uinput_init);
module_exit(fpga_uinput_exit);
