--- Title: DDC232x4.vhd
--- Description: 
---
---     o  0
---     | /       Copyright (c) 2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 11/11/2010
--- Version: 1.02
--- Revisions: 
---   1.00 - Initial release.
---   1.01 - Added a counting pattern test mode.
---   1.02 - Added a mechanism to automatically enable/disable the dvalid signal.
---          Protects against dvalidn's that are free-running or initially active.

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity DDC232x4 is
   port ( 
      -- EMIFA Interface
      i_eclk          : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;
      o_irq           : out std_logic;
      i_ilevel        : in  std_logic_vector(1 downto 0) := "00";
      i_ivector       : in  std_logic_vector(3 downto 0) := "0000";
      -- DDC232 Interface
      i_ddc_clk       : in  std_logic; -- 40 MHz converter clock
      o_ddc_conv      : out std_logic;
      i_ddc_dvalidn1  : in  std_logic;
      i_ddc_dvalidn2  : in  std_logic;
      i_ddc_dvalidn3  : in  std_logic;
      i_ddc_dvalidn4  : in  std_logic;
      o_ddc_dclk      : out std_logic;
      i_ddc_dout      : in  std_logic;
      o_ddc_resetn    : out std_logic;
      o_ddc_clk_cfg   : out std_logic;
      o_ddc_din_cfg1  : out std_logic;
      o_ddc_din_cfg2  : out std_logic;
      o_ddc_din_cfg3  : out std_logic;
      o_ddc_din_cfg4  : out std_logic;
	  
	  o_DSP_RD : out std_logic; --! Debug
	  o_FIFO_RD_ADDR_INC : out std_logic; --! Debug
	  o_FIFO_RD : out std_logic --! Debug
   );
end DDC232x4;

architecture rtl of DDC232x4 is

constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR(01, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR(02, 4);
constant CORE_ID:             std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR(99, 8);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR(10, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR(11, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR(11, 5);

component core_version
   port (
      clk           : in std_logic;                       -- system clock
      rd            : in std_logic;                       -- read enable
      ID            : in std_logic_vector(7 downto 0);    -- assigned ID number, 0xF0-0xFF are reserved for customers
      version_major : in std_logic_vector(3 downto 0);    -- major version number 1-15
      version_minor : in std_logic_vector(3 downto 0);    -- minor version number 0-15
      year          : in std_logic_vector(4 downto 0);    -- year since 2000
      month         : in std_logic_vector(3 downto 0);    -- month (1-12)
      day           : in std_logic_vector(4 downto 0);    -- day (1-32)
      ilevel        : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      ivector       : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)
      o_data        : out std_logic_vector(15 downto 0)
      );
end component;

component fixed20_to_float32
   port (
      a            : in  std_logic_vector(20 downto 0);
      operation_nd : in  std_logic;
      clk          : in  std_logic;
      result       : out std_logic_vector(31 downto 0);
      rdy          : out std_logic
   );
end component;

constant OFFSET_VERSION:      std_logic_vector(7 downto 0) := x"00";
constant OFFSET_CONTROL:      std_logic_vector(7 downto 0) := x"02";
constant OFFSET_CONFIG1:      std_logic_vector(7 downto 0) := x"04";
constant OFFSET_CONFIG2:      std_logic_vector(7 downto 0) := x"06";
constant OFFSET_CONFIG3:      std_logic_vector(7 downto 0) := x"08";
constant OFFSET_CONFIG4:      std_logic_vector(7 downto 0) := x"0A";
constant OFFSET_RAMADDR:      std_logic_vector(7 downto 0) := x"1E";
constant OFFSET_RAMDATAL:     std_logic_vector(7 downto 0) := x"20";
constant OFFSET_RAMDATAH:     std_logic_vector(7 downto 0) := x"22";
constant OFFSET_IRQCNTR : std_logic_vector(7 downto 0) := X"24";

signal ABus        : std_logic_vector(7 downto 0);
signal version_reg : std_logic_vector(15 downto 0);
signal ver_rd      : std_logic := '0';

signal cfg_dirty   : std_logic := '1';
signal wr_cfg_tgl  : std_logic := '0';
signal conv_enable : std_logic := '0';
signal conv_enable_r1 : std_logic := '0';
signal conv_enable_r2 : std_logic := '0';
signal conv_enable_r3 : std_logic := '0';
signal int_period  : std_logic_vector(13 downto 0) := conv_std_logic_vector(1614 * 4, 14);

signal config1  : std_logic_vector(11 downto 0) := x"FC0";
signal config2  : std_logic_vector(11 downto 0) := x"FC0";
signal config3  : std_logic_vector(11 downto 0) := x"FC0";
signal config4  : std_logic_vector(11 downto 0) := x"FC0";

signal test_mode : std_logic := '0';

constant RAM_ADDR_BITS : integer := 8;
signal ram_rd_addr : std_logic_vector(RAM_ADDR_BITS-1 downto 0) := (others=>'0');
signal ram_rd_data : std_logic_vector(31 downto 0) := (others=>'0');
signal ram_wr_addr : std_logic_vector(RAM_ADDR_BITS-1 downto 0) := (others=>'0');

type RAM_TYPE is array (0 to 2**RAM_ADDR_BITS-1) of std_logic_vector(31 downto 0);
signal ram : RAM_TYPE := (others=>(others=>'0'));

signal raddr_msb_r1 : std_logic := '1';
signal raddr_msb_r2 : std_logic := '1';
signal irq : std_logic := '0';

signal int_counter : std_logic_vector(13 downto 0) := conv_std_logic_vector(0, 14);

signal conv : std_logic := '0';
signal conv_r1 : std_logic := '0';
signal conv_r2 : std_logic := '0';

signal dvalid : std_logic := '0';
signal dvalid_r1 : std_logic := '0';
signal dvalid_r2 : std_logic := '0';
signal dvalid_en : std_logic := '0';

signal data_ready : std_logic := '0';

constant tSDCV : integer := 200; --TODO: change this back when we get a 40 MHz clock again --400; -- 10uS / (1/40MHz) = 400
signal tsdcv_counter : std_logic_vector(8 downto 0) := (others=>'1');

signal start_capture : std_logic := '0';

constant CAPTURE_TC : integer := 4 * 32 * 20; -- = 2560
signal capture_counter : std_logic_vector(11 downto 0) := conv_std_logic_vector(CAPTURE_TC, 12);

signal dclk : std_logic := '0';

signal shift_reg : std_logic_vector(19 downto 0) := (others=>'0');
signal bit_counter : std_logic_vector(4 downto 0) := (others=>'0');

signal f2f_a : std_logic_vector(20 downto 0) := (others => '0');

signal sample : std_logic_vector(19 downto 0) := (others=>'0');
signal sample_rdy : std_logic := '0';
signal sample_counter : std_logic_vector(23 downto 0) := (others=>'0');

signal float : std_logic_vector(31 downto 0) := (others=>'0');
signal float_rdy : std_logic := '0';

signal waddr_msb_r1 : std_logic := '0';
signal waddr_msb_r2 : std_logic := '0';
signal busy : std_logic := '0';

signal wr_cfg_r1 : std_logic := '0';
signal wr_cfg_r2 : std_logic := '0';
signal wr_cfg_r3 : std_logic := '0';

signal wr_cfg_flag : std_logic := '0';

constant tRST : integer := 40; -- 1us / (1/40MHz) = 40
constant tWTRST : integer := 80; -- 2us / (1/40MHz) = 80
constant CFG_BITS : integer := 12;
constant CFG_CNTR_TC : integer := tRST + tWTRST + (2 * CFG_BITS); -- 144
signal cfg_counter : std_logic_vector(7 downto 0) := (others=>'1');

signal ddc_resetn : std_logic := '1';
signal ddc_clk_cfg : std_logic := '0';
signal ddc_din_cfg1_sh : std_logic_vector(11 downto 0) := x"FC0";
signal ddc_din_cfg2_sh : std_logic_vector(11 downto 0) := x"FC0";
signal ddc_din_cfg3_sh : std_logic_vector(11 downto 0) := x"FC0";
signal ddc_din_cfg4_sh : std_logic_vector(11 downto 0) := x"FC0";

signal cfg_dirty_rst_tgl : std_logic := '0';
signal cfg_dirty_rst_r1  : std_logic := '0';
signal cfg_dirty_rst_r2  : std_logic := '0';

signal s_irq_cntr : std_logic_vector(15 downto 0) := (others => '0');

signal 	s_DSP_RD : std_logic := '0'; --! Debug
signal 	s_FIFO_RD_ADDR_INC : std_logic := '0'; --! Debug
signal 	s_FIFO_RD : std_logic := '0'; --! Debug



begin -- architecture : rtl

version : core_version
   port map(
      clk           => i_eclk,               -- system clock
      rd            => ver_rd,               -- read enable
      ID            => CORE_ID,              -- assigned ID number, 0xFF if unassigned
      version_major => CORE_VERSION_MAJOR,   -- major version number 1-15
      version_minor => CORE_VERSION_MINOR,   -- minor version number 0-15
      year          => CORE_YEAR,            -- year since 2000
      month         => CORE_MONTH,           -- month (1-12)
      day           => CORE_DAY,             -- day (1-31)
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg
      );

ABus <= '0' & i_ABus & '0';

--Debug 
o_DSP_RD <= s_DSP_RD;
o_FIFO_RD_ADDR_INC <= s_FIFO_RD_ADDR_INC;
o_FIFO_RD <= s_FIFO_RD;

-------------------------------------------------------------------------------
-- Handle Read Requests from the EMIF Bus
-------------------------------------------------------------------------------
reg_read : process (i_eclk)
begin
   if rising_edge(i_eclk) then
      ver_rd <= '0';

		s_FIFO_RD <= '0';
		s_DSP_RD <= '0';

      if i_cs = '0' then
         o_DBus <= (others=>'0');
      else
         case ABus is
            when OFFSET_VERSION  =>
               ver_rd <= i_rd_en;
               o_DBus <= version_reg;
            when OFFSET_CONTROL  =>
               o_DBus <= cfg_dirty & conv_enable & int_period;
            when OFFSET_CONFIG1  =>
               o_DBus <= x"0" & config1;
            when OFFSET_CONFIG2  =>
               o_DBus <= x"0" & config2;
            when OFFSET_CONFIG3  =>
               o_DBus <= x"0" & config3;
            when OFFSET_CONFIG4  =>
               o_DBus <= x"0" & config4;
            when OFFSET_RAMADDR  =>
               o_DBus <= test_mode & conv_std_logic_vector(0, 15-RAM_ADDR_BITS) & ram_rd_addr;
            when OFFSET_RAMDATAL =>
               o_DBus <= ram_rd_data(15 downto 0);
			   s_FIFO_RD <= '1';
            when OFFSET_RAMDATAH =>
               o_DBus <= ram_rd_data(31 downto 16);
			   s_FIFO_RD <= '1';
			when OFFSET_IRQCNTR =>
			   o_DBus <= s_irq_cntr;
            when others => null;
         end case;
		 
		 s_DSP_RD <= '1';
      end if;
   end if;
end process reg_read;
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- Handle Write Requests from the EMIF Bus
-------------------------------------------------------------------------------
reg_write : process (i_eclk)
begin
   if rising_edge(i_eclk) then
      if i_cs = '1' and i_wr_en = '1' then
         case ABus is
            when OFFSET_CONTROL  =>
               if i_DBus(15) = '1' and i_DBus(14) = '0' then
                  wr_cfg_tgl <= '1';--not wr_cfg_tgl;
               end if;
               conv_enable <= i_DBus(14);
               int_period  <= i_DBus(13 downto 0);
            when OFFSET_CONFIG1  =>
               config1(11 downto 0) <= i_DBus(11 downto 0);
               --config1(0)           <= i_DBus(0);
            when OFFSET_CONFIG2  =>
               config2(11 downto 0) <= i_DBus(11 downto 0);
               --config2(0)           <= i_DBus(0);
            when OFFSET_CONFIG3  =>
               config3(11 downto 0) <= i_DBus(11 downto 0);
               --config3(0)           <= i_DBus(0);
            when OFFSET_CONFIG4  =>
               config4(11 downto 0) <= i_DBus(11 downto 0);
               --config4(0)           <= i_DBus(0);
            when OFFSET_RAMADDR  =>
               test_mode <= i_DBus(15);
            when others => null;
         end case;
      end if;
		
		if (ddc_resetn = '0') then
			wr_cfg_tgl <= '0';
		end if;
         
      cfg_dirty_rst_r1 <= cfg_dirty_rst_tgl;
      cfg_dirty_rst_r2 <= cfg_dirty_rst_r1;
      
      -- handle setting/resetting of cfg_dirty flag
      if i_cs = '1' and i_wr_en = '1' and
         (ABus = OFFSET_CONFIG1 or
          ABus = OFFSET_CONFIG2 or
          ABus = OFFSET_CONFIG3 or
          ABus = OFFSET_CONFIG4) then
         -- set the flag
         cfg_dirty <= '1';
      elsif cfg_dirty_rst_r1 /= cfg_dirty_rst_r2 then
         -- reset the flag
         cfg_dirty <= '0';
      end if;
   end if;
end process reg_write;
-------------------------------------------------------------------------------

ram_rd_addr(RAM_ADDR_BITS-1) <= not ram_wr_addr(RAM_ADDR_BITS-1);

-------------------------------------------------------------------------------
-- EMIF RAM Read Process
-------------------------------------------------------------------------------
ram_rd_side : process (i_eclk)
begin
   if rising_edge(i_eclk) then
      ram_rd_data <= ram(conv_integer(ram_rd_addr));

	  s_FIFO_RD_ADDR_INC <= '0';

      if i_cs = '1' then
         -- if the RAM address register is written...
         if i_wr_en = '1' and ABus = OFFSET_RAMADDR then
            -- store the new address
            ram_rd_addr(RAM_ADDR_BITS-2 downto 0) <= i_DBus(RAM_ADDR_BITS-2 downto 0);
         -- if the RAM data (high) register is read...
         elsif i_rd_en = '1' and ABus = OFFSET_RAMDATAH then
            -- increment the address (all except the MSB, which is controlled by logic)
            ram_rd_addr(RAM_ADDR_BITS-2 downto 0) <= ram_rd_addr(RAM_ADDR_BITS-2 downto 0) + '1';
			
			s_FIFO_RD_ADDR_INC <= '1';
         end if;
      end if;
   end if;
end process ram_rd_side;
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- Ping Pong Buffer Interrupt Process
-------------------------------------------------------------------------------
interrupt_gen : process (i_eclk)
begin
   if rising_edge(i_eclk) then
      raddr_msb_r1 <= ram_rd_addr(RAM_ADDR_BITS-1);
      raddr_msb_r2 <= raddr_msb_r1;
      
      -- if RAM address MSB toggles
      if raddr_msb_r1 /= raddr_msb_r2 then
         -- set the interrupt line
         irq <= '1';
		 
		 s_irq_cntr <= s_irq_cntr + '1';
      -- if the RAM data (high) register is read...
      elsif i_rd_en = '1' and ABus = OFFSET_RAMDATAH then
         -- reset the interrupt line
         irq <= '0';
      end if;
   end if;
end process;
-------------------------------------------------------------------------------

o_irq <= irq;

-------------------------------------------------------------------------------
-- Generate conv Signal (based on int_counter) Process
-------------------------------------------------------------------------------
gen_conv : process (i_ddc_clk)
begin
   if rising_edge(i_ddc_clk) then
		-- Clock domain crossing to lower chances of metastability
		conv_enable_r1 <= conv_enable;
		conv_enable_r2 <= conv_enable_r1;
		conv_enable_r3 <= conv_enable_r2;
	
      -- if integration counter is at its terminal count
      if int_counter >= int_period then
         -- if converting is enabled or we need to complete the 2nd half of cycle
         if conv = '1' or conv_enable_r2 = '1' then
            -- reset the integration counter and toggle conv
            int_counter <= conv_std_logic_vector(1, 14);
            conv <= not conv;
         end if;
      else
         int_counter <= int_counter + '1';
      end if;
   end if;
end process gen_conv;
-------------------------------------------------------------------------------

dvalid <= not (i_ddc_dvalidn1 and i_ddc_dvalidn2 and i_ddc_dvalidn3 and i_ddc_dvalidn4);

-- Mike's most recent changes... merg in later...
-- Last external dvalidn in triggers the internal dvalid.-- Also triggered by activation of dvalid_en if dvalidn's were already active.
-- Also triggered by activation of dvalid_en if dvalidn's were already active.
--dvalid <= dvalid_en and not (i_ddc_dvalidn1 or i_ddc_dvalidn2 or i_ddc_dvalidn3 or i_ddc_dvalidn4); 

-------------------------------------------------------------------------------
-- DDC Data Capture to Fixed to Float Conversion Process
-------------------------------------------------------------------------------
data_capture : process (i_ddc_clk)
begin
   if rising_edge(i_ddc_clk) then
      dvalid_r1 <= dvalid;
      dvalid_r2 <= dvalid_r1;
      
      start_capture <= '0';
      sample_rdy <= '0';
      dclk <= '0';
      
      if conv_r1 = '1' and conv_r2 = '0' then
         -- set dvalid_en when conversion cycle has begun
         dvalid_en <= '1';
      elsif dvalid_r1 = '1' and dvalid_r2 = '0' and conv = '0' and conv_enable_r2 = '0' then
         -- reset dvalid_en when dvalid detected during 2nd half of cycle and conv's have been disabled
         dvalid_en <= '0';
      end if;
      
      if dvalid_r1 = '1' and dvalid_r2 = '0' then
         -- set data_ready when dvalid detected
         data_ready <= '1';
      elsif data_ready = '1' and int_counter >= int_period then
         -- reset data_ready when conv toggles (integration period reached)
         data_ready <= '0';
      end if;
      
      if data_ready = '1' and int_counter >= int_period then
         -- reset wait counter when conv toggles(integration period reached)
         tsdcv_counter <= (others=>'0');
      elsif tsdcv_counter < conv_std_logic_vector(tSDCV, 9) then
         -- increment wait counter
         tsdcv_counter <= tsdcv_counter + '1';
      elsif tsdcv_counter = conv_std_logic_vector(tSDCV, 9) then
         -- hold wait counter, and pulse start_capture when terminal count reached
         tsdcv_counter <= (others=>'1');
         start_capture <= '1';
      end if;
      
      if start_capture = '1' then
         capture_counter <= (others=>'0');
		 bit_counter <= "00000";
      elsif capture_counter < conv_std_logic_vector(CAPTURE_TC, 12) then
         dclk <= not dclk;
         if dclk = '0' then
            shift_reg <= shift_reg(18 downto 0) & i_ddc_dout;
            -- modulus counter for 20 bits of data
            if bit_counter = "10011" then
               bit_counter <= "00000";
            else
               bit_counter <= bit_counter + '1';
            end if;
				
			capture_counter <= capture_counter;
         elsif (dclk = '1') then
            capture_counter <= capture_counter + '1';
            -- grab the sample from the shift register
            if bit_counter = "00000" then
               sample <= shift_reg;
               sample_rdy <= '1';
            end if;
         end if;
      end if;
   end if;
end process data_capture;
-------------------------------------------------------------------------------

fixed2float : fixed20_to_float32
   port map (
      a            => f2f_a,--"0"&sample,
      operation_nd => sample_rdy,
      clk          => i_ddc_clk,
      result       => float,
      rdy          => float_rdy
   );

f2f_a <= "0"&sample;

-------------------------------------------------------------------------------
-- Write Floating Point Data to RAM Process
-------------------------------------------------------------------------------
ram_wr_side : process (i_ddc_clk)
begin
   if rising_edge(i_ddc_clk) then
	--	if (ddc_resetn = '0') then
	--		-- Force to 256 BRAM boundary on reset
	--		-- TODO: This should be temporary, because we need to figure out why we are sometimes off by 128 samples...
	--		ram_wr_addr(RAM_ADDR_BITS-2) <= '0';
    --  els
	  if start_capture = '1' then
         -- make sure the starting address is correct
			-- Reset the ram_wr_addr start address at the beginning of a new capture, or if a reset occurs
         ram_wr_addr(RAM_ADDR_BITS-2 downto 0) <= (others => '1');
      elsif float_rdy = '1' then
         -- count from 127 downto 0, then 255 downto 128, then flip the MSB (which adds or subs 256)
         ram_wr_addr(RAM_ADDR_BITS-2 downto 0) <= ram_wr_addr(RAM_ADDR_BITS-2 downto 0) - '1';
         
         if ram_wr_addr(RAM_ADDR_BITS-2 downto 0) = conv_std_logic_vector( 0, RAM_ADDR_BITS-1) then--"10000000" then
            ram_wr_addr(RAM_ADDR_BITS-1) <= not ram_wr_addr(RAM_ADDR_BITS-1);
         end if;
      end if;

      if float_rdy = '1' then
         if ram_wr_addr(RAM_ADDR_BITS-2 downto 0) = conv_std_logic_vector(0, RAM_ADDR_BITS-1) then
            -- increment the sample counter for test mode
            sample_counter <= sample_counter + '1';
         end if;
         
         -- check if counting pattern test mode is enabled
         if test_mode = '0' then
            ram(conv_integer(ram_wr_addr)) <= float;
         else -- test_mode = '1'
            ram(conv_integer(ram_wr_addr)) <= ram_wr_addr(RAM_ADDR_BITS-1 downto 0) & sample_counter;
         end if;
      end if;
   end if;
end process ram_wr_side;
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- Generate DDC Busy Signal (When conv_en is low and current ping pong side is filled) Process
-------------------------------------------------------------------------------
gen_busy : process (i_ddc_clk)
begin
   if rising_edge(i_ddc_clk) then
      conv_r1 <= conv;
      conv_r2 <= conv_r1;
      
      waddr_msb_r1 <= ram_wr_addr(RAM_ADDR_BITS-1);
      waddr_msb_r2 <= waddr_msb_r1;
      
      -- generate a busy flag...
      -- if conversion cycle has begun
      if (conv_r1 = '1' and conv_r2 = '0') then
         busy <= '1'; -- set busy
      -- if conversion is disabled, and not in the middle of a cycle
      elsif conv_enable_r2 = '0' and conv = '0' and ram_wr_addr(RAM_ADDR_BITS-2 downto 0) = conv_std_logic_vector( 0, RAM_ADDR_BITS-1) then --waddr_msb_r1 /= waddr_msb_r2 then
         busy <= '0'; -- reset busy when RAM write address is flipped, indicating data ready
      end if;
   end if;
end process gen_busy;
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- Write Configuration to DDC (waiting until DDC is not busy) Process
-------------------------------------------------------------------------------
config_write : process (i_ddc_clk)
begin
   if rising_edge(i_ddc_clk) then
      wr_cfg_r1 <= wr_cfg_tgl;
      wr_cfg_r2 <= wr_cfg_r1;
      wr_cfg_r3 <= wr_cfg_r2;
		
		if (wr_cfg_r2 = '1' and wr_cfg_r3 = '0') then
			wr_cfg_flag <= '1';
		end if;
		
      -- test for start conditions
      if wr_cfg_flag = '1' and busy = '0' then
         cfg_counter <= (others=>'0');
			wr_cfg_flag <= '0';
      elsif cfg_counter < conv_std_logic_vector(CFG_CNTR_TC, 8) then
         cfg_counter <= cfg_counter + '1';
      end if;
      
      -- drive reset
      if cfg_counter < conv_std_logic_vector(tRST, 8) then
         ddc_resetn <= '0';
      else
         ddc_resetn <= '1';
      end if;
      
      if cfg_counter < conv_std_logic_vector(CFG_CNTR_TC, 8) and
         cfg_counter >= conv_std_logic_vector(tRST + tWTRST, 8) then
         -- clock out 12 config bits
         ddc_clk_cfg <= not ddc_clk_cfg;
         
         if ddc_clk_cfg = '0' then
            ddc_din_cfg1_sh <= ddc_din_cfg1_sh(10 downto 0) & ddc_din_cfg1_sh(11);
            ddc_din_cfg2_sh <= ddc_din_cfg2_sh(10 downto 0) & ddc_din_cfg2_sh(11);
            ddc_din_cfg3_sh <= ddc_din_cfg3_sh(10 downto 0) & ddc_din_cfg3_sh(11);
            ddc_din_cfg4_sh <= ddc_din_cfg4_sh(10 downto 0) & ddc_din_cfg4_sh(11);
         end if;
      else
         ddc_clk_cfg <= '0';
         ddc_din_cfg1_sh <= config1;
         ddc_din_cfg2_sh <= config2;
         ddc_din_cfg3_sh <= config3;
         ddc_din_cfg4_sh <= config4;
      end if;
      
      -- trigger reset of the cfg_dirty flag
      if cfg_counter = conv_std_logic_vector(CFG_CNTR_TC-1, 8) then
         cfg_dirty_rst_tgl <= not cfg_dirty_rst_tgl;
			
			-- Make sure that all ping pong buffers and such as reset after configuration
      end if;
   end if;
end process config_write;
-------------------------------------------------------------------------------

o_ddc_conv <= conv;
o_ddc_dclk <= dclk;
o_ddc_resetn   <= ddc_resetn;
o_ddc_clk_cfg  <= ddc_clk_cfg;
o_ddc_din_cfg1 <= ddc_din_cfg1_sh(0);
o_ddc_din_cfg2 <= ddc_din_cfg2_sh(0);
o_ddc_din_cfg3 <= ddc_din_cfg3_sh(0);
o_ddc_din_cfg4 <= ddc_din_cfg4_sh(0);

end rtl;
