--- Title: lcd_serdes_nl8060BC16-01.vhd
--- Description:  Serializer for NEC based display (800x600) direct interface.
---
---     o  0
---     | /       Copyright (c) 2011
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 01/21/2011
--- Version: 1.00
-------------------------------------------------------------------------------
--- Rev. Date     Eng Description
--- ---- -------- --- ---------------------------------------------------------
--- 1.00 01/21/11 MAW Initial version.
--- 1.01 02/27/12 Add better hooks for resetting DCM and FIFO counters
-------------------------------------------------------------------------------

library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;

library WORK;
use WORK.MityDSP_L138_pkg.ALL;

entity lcd_serdes_nl8060BC16_01 is
   generic (
      MAPPING : string := "6BIT" -- "A" or "B" or "6BIT"
   );
   port (
      
      -- LCD Controller interface (from OMAP-L138)
      i_lcd_d       : in std_logic_vector(15 downto 0);
      i_lcd_hsync   : in std_logic;
      i_lcd_vsync   : in std_logic;
      i_lcd_mclk    : in std_logic;
      i_lcd_pclk    : in std_logic;  -- ~38 MHz
      i_lcd_enb     : in std_logic;
      
      -- DCM reset
      i_dcm_reset   : std_logic := '0';
            
      -- LvDS I/O pins for LCD and some control signals
      o_serdes_clk   : out std_logic;
      o_serdes_a0     : out std_logic;
      o_serdes_a1     : out std_logic;
      o_serdes_a2     : out std_logic;
      o_serdes_a3     : out std_logic -- not used in 6 bit mode
      
   );
end lcd_serdes_nl8060BC16_01;

--------------------------------------------------------------------------
-- ARCHITECTURE
--------------------------------------------------------------------------
architecture rtl of lcd_serdes_nl8060BC16_01 is

signal lcd_r, lcd_g, lcd_b : std_logic_vector(7 downto 0) := "00000000";
signal lcd_r_r, lcd_g_r, lcd_b_r : std_logic_vector(7 downto 0) := "00000000";
signal lcd_a0_sh, lcd_a1_sh, lcd_a2_sh, lcd_a3_sh : std_logic_vector(6 downto 0) := "0000000";
signal lcd_de, lcd_de_r : std_logic := '0';
signal serdes_sh : std_logic_vector(6 downto 0) := "0000001";
signal serdes_clk : std_logic := '0';
signal ser_clk : std_logic;
signal pclk : std_logic;

signal ser_fifo_re : std_logic := '0';
signal data_dup_sh : std_logic_vector(1 downto 0) := "01";
signal lcd_clk_sh : std_logic_vector(1 downto 0) := "10";

signal write_addr : std_logic_vector(4 downto 0) := "00000";
signal read_addr  : std_logic_vector(4 downto 0) := "10000";

signal dcm_lock : std_logic;
signal dcm_status : std_logic_vector(7 downto 0);

signal fifo_din, fifo_dout : std_logic_vector(20 downto 0);
signal lcd_clk_inv : std_logic;

begin -- architecture: rtl

DCM_Serdes_Clkgen : DCM_SP
    generic map (
      CLKDV_DIVIDE => 2.0,                   -- CLKDV divide value
                                             -- (1.5,2,2.5,3,3.5,4,4.5,5,5.5,6,6.5,7,7.5,8,9,10,11,12,13,14,15,16).
      CLKFX_DIVIDE => 1,                     -- Divide value - D - (1-32)
      CLKFX_MULTIPLY => 7,                   -- Multiply value - M - (2-32)
      CLKIN_DIVIDE_BY_2 => FALSE,            -- CLKIN divide by two (TRUE/FALSE)
      CLKIN_PERIOD => 25.0,                  -- Input clock period specified in nS
      CLKOUT_PHASE_SHIFT => "NONE",          -- Output phase shift (NONE, FIXED, VARIABLE)
      CLK_FEEDBACK => "1X",                  -- Feedback source (NONE, 1X, 2X)
      DESKEW_ADJUST => "SYSTEM_SYNCHRONOUS", -- SYSTEM_SYNCHRNOUS or SOURCE_SYNCHRONOUS
      DFS_FREQUENCY_MODE => "LOW",           -- Unsupported - Do not change value
      DLL_FREQUENCY_MODE => "LOW",           -- Unsupported - Do not change value
      DSS_MODE => "NONE",                    -- Unsupported - Do not change value
      DUTY_CYCLE_CORRECTION => TRUE,         -- Unsupported - Do not change value
      FACTORY_JF => X"c080",                 -- Unsupported - Do not change value
      PHASE_SHIFT => 0,                      -- Amount of fixed phase shift (-255 to 255)
      STARTUP_WAIT => FALSE                  -- Delay config DONE until DCM LOCKED (TRUE/FALSE)
   )
   port map (
      CLK0 => pclk,         -- 1-bit 0 degree clock output
      CLK180 => open,       -- 1-bit 180 degree clock output
      CLK270 => open,       -- 1-bit 270 degree clock output
      CLK2X => open,        -- 1-bit 2X clock frequency clock output
      CLK2X180 => open,     -- 1-bit 2X clock frequency, 180 degree clock output
      CLK90 => open,        -- 1-bit 90 degree clock output
      CLKDV => open,        -- 1-bit Divided clock output
      CLKFX => serdes_clk,     -- 1-bit Digital Frequency Synthesizer output (DFS)
      CLKFX180 => open,     -- 1-bit 180 degree CLKFX output
      LOCKED => dcm_lock,   -- 1-bit DCM Lock Output
      PSDONE => open,       -- 1-bit Phase shift done output
      STATUS => dcm_status, -- 8-bit DCM status output
      CLKFB => pclk,        -- 1-bit Clock feedback input
      CLKIN => lcd_clk_inv,  -- 1-bit Clock input (the omap is delivering it inverted)
      DSSEN => open,        -- 1-bit Unsupported
      PSCLK => '0',         -- 1-bit Phase shift clock input
      PSEN => '0',          -- 1-bit Phase shift enable
      PSINCDEC => '0',      -- 1-bit Phase shift increment/decrement input
      RST => i_dcm_reset    -- 1-bit Active high reset input
   );

lcd_clk_inv <= not i_lcd_pclk;

FIFOBLOCK : for i in 0 to 20 generate
begin
RAM32x1FIFO : RAM32X1D
   generic map ( INIT => X"00000000" )
   port map (
       DPO => fifo_dout(i),
       SPO => open,
       A0  => write_addr(0),
       A1  => write_addr(1),
       A2  => write_addr(2),
       A3  => write_addr(3),
       A4  => write_addr(4),
       D   => fifo_din(i),
       DPRA0 => read_addr(0),
       DPRA1 => read_addr(1),
       DPRA2 => read_addr(2),
       DPRA3 => read_addr(3),
       DPRA4 => read_addr(4),
       WCLK  => pclk,
       WE    => '1' );    
end generate;

fifo_din(5 downto 0)   <= i_lcd_d(4 downto 0) & '0';
fifo_din(11 downto 6)  <= i_lcd_d(10 downto 5);
fifo_din(17 downto 12) <= i_lcd_d(15 downto 11) & '0';
fifo_din(18)           <= i_lcd_enb;

lcd_b(7 downto 2) <= fifo_dout(5 downto 0);
lcd_g(7 downto 2) <= fifo_dout(11 downto 6);
lcd_r(7 downto 2) <= fifo_dout(17 downto 12);
lcd_de <= fifo_dout(18);

writeaddr : process(pclk)
begin
    if dcm_lock='0' then
        write_addr <= "00000";
    elsif rising_edge(pclk) then
        write_addr <= write_addr+'1';
    end if;
end process writeaddr;

readaddr : process(serdes_clk)
begin
    if dcm_lock='0' then
       read_addr <= "10000";
    elsif rising_edge(serdes_clk) then
       if ser_fifo_re='1' then
           read_addr <= read_addr+'1';
       end if;
    end if;
end process readaddr;

o_serdes_clk <= ser_clk;
o_serdes_a0 <= lcd_a0_sh(6);
o_serdes_a1 <= lcd_a1_sh(6);
o_serdes_a2 <= lcd_a2_sh(6);
o_serdes_a3 <= lcd_a3_sh(6);

shift_clk : process (serdes_clk)
begin
   if rising_edge(serdes_clk) then
            
      if serdes_sh(0) = '1' then
         ser_fifo_re <= '1';
         lcd_r_r <= lcd_r;
         lcd_g_r <= lcd_g;
         lcd_b_r <= lcd_b;
         lcd_de_r <= lcd_de;
      else
         ser_fifo_re <= '0';   
      end if;

      serdes_sh <= serdes_sh(5 downto 0) & serdes_sh(6);
      
      if serdes_sh(2) = '1' then
         ser_clk <= '0';
      elsif serdes_sh(5) = '1' then
         ser_clk <= '1';
      end if;
   end if;
end process shift_clk;

mapping_a : if MAPPING="A" generate
begin

shiftdata_a : process(serdes_clk)
begin
   if rising_edge(serdes_clk) then
      --The SerDes receiver expects the D(0) first, then D(1)..D(6)
      if serdes_sh(0) = '1' then
         lcd_a0_sh(0) <= lcd_r_r(2);
         lcd_a0_sh(1) <= lcd_r_r(3); 
         lcd_a0_sh(2) <= lcd_r_r(4);
         lcd_a0_sh(3) <= lcd_r_r(5);
         lcd_a0_sh(4) <= lcd_r_r(6);
         lcd_a0_sh(5) <= lcd_r_r(7);
         lcd_a0_sh(6) <= lcd_g_r(2);

         lcd_a1_sh(0) <= lcd_g_r(3);
         lcd_a1_sh(1) <= lcd_g_r(4);
         lcd_a1_sh(2) <= lcd_g_r(5);
         lcd_a1_sh(3) <= lcd_g_r(6);
         lcd_a1_sh(4) <= lcd_g_r(7);
         lcd_a1_sh(5) <= lcd_b_r(2);
         lcd_a1_sh(6) <= lcd_b_r(3);

         lcd_a2_sh(0) <= lcd_b_r(4);
         lcd_a2_sh(1) <= lcd_b_r(5);
         lcd_a2_sh(2) <= lcd_b_r(6);
         lcd_a2_sh(3) <= lcd_b_r(7);
         lcd_a2_sh(4) <= '0'; -- (after LCD ON)
         lcd_a2_sh(5) <= '0'; -- LCD ON (after reset)
         lcd_a2_sh(6) <= lcd_de_r;

         lcd_a3_sh(0) <= lcd_r_r(0);
         lcd_a3_sh(1) <= lcd_r_r(1);
         lcd_a3_sh(2) <= lcd_g_r(0);
         lcd_a3_sh(3) <= lcd_g_r(1);
         lcd_a3_sh(4) <= lcd_b_r(0); -- (after LCD ON)
         lcd_a3_sh(5) <= lcd_b_r(1); -- LCD ON (after reset)
         lcd_a3_sh(6) <= '0';

      else
         lcd_a0_sh <= lcd_a0_sh(5 downto 0) & lcd_a0_sh(6);
         lcd_a1_sh <= lcd_a1_sh(5 downto 0) & lcd_a1_sh(6);
         lcd_a2_sh <= lcd_a2_sh(5 downto 0) & lcd_a2_sh(6);
         lcd_a3_sh <= lcd_a3_sh(5 downto 0) & lcd_a3_sh(6);
      end if;
   end if;
end process shiftdata_a;

end generate mapping_a;

mapping_b : if MAPPING="B" generate
begin

shiftdata_b : process(serdes_clk)
begin
   if rising_edge(serdes_clk) then
      --The SerDes receiver expects the D(0) first, then D(1)..D(6)
      if serdes_sh(0) = '1' then
         lcd_a0_sh(0) <= lcd_r_r(0);
         lcd_a0_sh(1) <= lcd_r_r(1); 
         lcd_a0_sh(2) <= lcd_r_r(2);
         lcd_a0_sh(3) <= lcd_r_r(3);
         lcd_a0_sh(4) <= lcd_r_r(4);
         lcd_a0_sh(5) <= lcd_r_r(5);
         lcd_a0_sh(6) <= lcd_g_r(0);

         lcd_a1_sh(0) <= lcd_g_r(1);
         lcd_a1_sh(1) <= lcd_g_r(2);
         lcd_a1_sh(2) <= lcd_g_r(3);
         lcd_a1_sh(3) <= lcd_g_r(4);
         lcd_a1_sh(4) <= lcd_g_r(5);
         lcd_a1_sh(5) <= lcd_b_r(0);
         lcd_a1_sh(6) <= lcd_b_r(1);

         lcd_a2_sh(0) <= lcd_b_r(2);
         lcd_a2_sh(1) <= lcd_b_r(3);
         lcd_a2_sh(2) <= lcd_b_r(4);
         lcd_a2_sh(3) <= lcd_b_r(5);
         lcd_a2_sh(4) <= '0'; -- (after LCD ON)
         lcd_a2_sh(5) <= '0'; -- LCD ON (after reset)
         lcd_a2_sh(6) <= lcd_de_r;

         lcd_a3_sh(0) <= lcd_r_r(6);
         lcd_a3_sh(1) <= lcd_r_r(7);
         lcd_a3_sh(2) <= lcd_g_r(6);
         lcd_a3_sh(3) <= lcd_g_r(7);
         lcd_a3_sh(4) <= lcd_b_r(6); -- (after LCD ON)
         lcd_a3_sh(5) <= lcd_b_r(7); -- LCD ON (after reset)
         lcd_a3_sh(6) <= '0';

      else
         lcd_a0_sh <= lcd_a0_sh(5 downto 0) & lcd_a0_sh(6);
         lcd_a1_sh <= lcd_a1_sh(5 downto 0) & lcd_a1_sh(6);
         lcd_a2_sh <= lcd_a2_sh(5 downto 0) & lcd_a2_sh(6);
         lcd_a3_sh <= lcd_a3_sh(5 downto 0) & lcd_a3_sh(6);
      end if;
   end if;
end process shiftdata_b;

end generate mapping_b;

mapping_6bit : if MAPPING="6BIT" generate
begin

shiftdata_6bit : process(serdes_clk)
begin
   if rising_edge(serdes_clk) then
      --The SerDes receiver expects the D(0) first, then D(1)..D(6)
      if serdes_sh(0) = '1' then
         lcd_a0_sh(0) <= lcd_r_r(2);
         lcd_a0_sh(1) <= lcd_r_r(3); 
         lcd_a0_sh(2) <= lcd_r_r(4);
         lcd_a0_sh(3) <= lcd_r_r(5);
         lcd_a0_sh(4) <= lcd_r_r(6);
         lcd_a0_sh(5) <= lcd_r_r(7);
         lcd_a0_sh(6) <= lcd_g_r(2);

         lcd_a1_sh(0) <= lcd_g_r(3);
         lcd_a1_sh(1) <= lcd_g_r(4);
         lcd_a1_sh(2) <= lcd_g_r(5);
         lcd_a1_sh(3) <= lcd_g_r(6);
         lcd_a1_sh(4) <= lcd_g_r(7);
         lcd_a1_sh(5) <= lcd_b_r(2);
         lcd_a1_sh(6) <= lcd_b_r(3);

         lcd_a2_sh(0) <= lcd_b_r(4);
         lcd_a2_sh(1) <= lcd_b_r(5);
         lcd_a2_sh(2) <= lcd_b_r(6);
         lcd_a2_sh(3) <= lcd_b_r(7);
         lcd_a2_sh(4) <= '0'; -- (after LCD ON)
         lcd_a2_sh(5) <= '0'; -- LCD ON (after reset)
         lcd_a2_sh(6) <= lcd_de_r;

      else
         lcd_a0_sh <= lcd_a0_sh(5 downto 0) & lcd_a0_sh(6);
         lcd_a1_sh <= lcd_a1_sh(5 downto 0) & lcd_a1_sh(6);
         lcd_a2_sh <= lcd_a2_sh(5 downto 0) & lcd_a2_sh(6);
      end if;
   end if;
end process shiftdata_6bit;

end generate mapping_6bit;

end rtl;
