library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity DDC232_tb is
end DDC232_tb;

architecture sim of DDC232_tb is

component DDC232_sim
   port (
      clk      : in  std_logic; -- assuming 4x input clock mode
      conv     : in  std_logic;
      din_cfg  : in  std_logic;
      clk_cfg  : in  std_logic;
      reset_n  : in  std_logic;
      dvalid_n : out std_logic;
      dclk     : in  std_logic;
      dout     : out std_logic;
      din      : in  std_logic
   );
end component;

component DDC232x4
   port ( 
      -- EMIFA Interface
      i_eclk          : in  std_logic;
      i_ABus          : in  std_logic_vector(5 downto 0);
      i_DBus          : in  std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in  std_logic;
      i_rd_en         : in  std_logic;
      i_cs            : in  std_logic;
      o_irq           : out std_logic;
      i_ilevel        : in  std_logic_vector(1 downto 0) := "00";
      i_ivector       : in  std_logic_vector(3 downto 0) := "0000";
      -- DDC232 Interface
      i_ddc_clk       : in  std_logic; -- 40 MHz converter clock
      o_ddc_conv      : out std_logic;
      i_ddc_dvalidn1  : in  std_logic;
      i_ddc_dvalidn2  : in  std_logic;
      i_ddc_dvalidn3  : in  std_logic;
      i_ddc_dvalidn4  : in  std_logic;
      o_ddc_dclk      : out std_logic;
      i_ddc_dout      : in  std_logic;
      o_ddc_resetn    : out std_logic;
      o_ddc_clk_cfg   : out std_logic;
      o_ddc_din_cfg1  : out std_logic;
      o_ddc_din_cfg2  : out std_logic;
      o_ddc_din_cfg3  : out std_logic;
      o_ddc_din_cfg4  : out std_logic
   );
end component;

constant DATA_BITS : integer := 32 * 20;

signal eclk : std_logic := '0';
signal addr : std_logic_vector(5 downto 0) := "000000";
signal din  : std_logic_vector(15 downto 0) := x"0000";
signal dout : std_logic_vector(15 downto 0);
signal wr   : std_logic := '0';
signal rd   : std_logic := '0';
signal cs   : std_logic := '0';
signal irq  : std_logic;

signal ddc_clk      : std_logic := '0';
signal ddc_conv     : std_logic := '0';
signal ddc_dvalidn1 : std_logic;
signal ddc_dvalidn2 : std_logic;
signal ddc_dvalidn3 : std_logic;
signal ddc_dvalidn4 : std_logic;
signal ddc_dclk     : std_logic := '0';
signal ddc_dout1    : std_logic;
signal ddc_dout2    : std_logic;
signal ddc_dout3    : std_logic;
signal ddc_dout4    : std_logic;
signal ddc_din1     : std_logic;
signal ddc_din2     : std_logic;
signal ddc_din3     : std_logic;
signal ddc_din4     : std_logic;
signal ddc_resetn   : std_logic;
signal ddc_clk_cfg  : std_logic;
signal ddc_din_cfg1 : std_logic;
signal ddc_din_cfg2 : std_logic;
signal ddc_din_cfg3 : std_logic;
signal ddc_din_cfg4 : std_logic;

signal datal : std_logic_vector(15 downto 0) := x"0000";
signal data  : std_logic_vector(31 downto 0) := x"00000000";

begin -- architecture : sim

ddc_din1 <= '0';

DDC232_1 : DDC232_sim
   port map (
      clk      => ddc_clk,
      conv     => ddc_conv,
      din_cfg  => '0',
      clk_cfg  => '0',
      reset_n  => '1',
      dvalid_n => ddc_dvalidn1,
      dclk     => ddc_dclk,
      dout     => ddc_dout1,
      din      => ddc_din1
   );

ddc_din2 <= ddc_dout1;

DDC232_2 : DDC232_sim
   port map (
      clk      => ddc_clk,
      conv     => ddc_conv,
      din_cfg  => '0',
      clk_cfg  => '0',
      reset_n  => '1',
      dvalid_n => ddc_dvalidn2,
      dclk     => ddc_dclk,
      dout     => ddc_dout2,
      din      => ddc_din2
   );

ddc_din3 <= ddc_dout2;

DDC232_3 : DDC232_sim
   port map (
      clk      => ddc_clk,
      conv     => ddc_conv,
      din_cfg  => '0',
      clk_cfg  => '0',
      reset_n  => '1',
      dvalid_n => ddc_dvalidn3,
      dclk     => ddc_dclk,
      dout     => ddc_dout3,
      din      => ddc_din3
   );

ddc_din4 <= ddc_dout3;

DDC232_4 : DDC232_sim
   port map (
      clk      => ddc_clk,
      conv     => ddc_conv,
      din_cfg  => '0',
      clk_cfg  => '0',
      reset_n  => '1',
      dvalid_n => ddc_dvalidn4,
      dclk     => ddc_dclk,
      dout     => ddc_dout4,
      din      => ddc_din4
   );

DUT : DDC232x4
   port map (
      i_eclk          => eclk,
      i_ABus          => addr,
      i_DBus          => din,
      o_DBus          => dout,
      i_wr_en         => wr,
      i_rd_en         => rd,
      i_cs            => cs,
      o_irq           => irq,
      i_ilevel        => "00",
      i_ivector       => "0000",
      i_ddc_clk       => ddc_clk,
      o_ddc_conv      => ddc_conv,
      i_ddc_dvalidn1  => ddc_dvalidn1,
      i_ddc_dvalidn2  => ddc_dvalidn2,
      i_ddc_dvalidn3  => ddc_dvalidn3,
      i_ddc_dvalidn4  => ddc_dvalidn4,
      o_ddc_dclk      => ddc_dclk,
      i_ddc_dout      => ddc_dout4,
      o_ddc_resetn    => ddc_resetn,
      o_ddc_clk_cfg   => ddc_clk_cfg,
      o_ddc_din_cfg1  => ddc_din_cfg1,
      o_ddc_din_cfg2  => ddc_din_cfg2,
      o_ddc_din_cfg3  => ddc_din_cfg3,
      o_ddc_din_cfg4  => ddc_din_cfg4
   );

eclk <= not eclk after 5 ns; -- 100 MHz
ddc_clk <= not ddc_clk after 12.5 ns; -- 40 MHz

bus_drive : process
variable conv_cntr : integer;
variable readout_cntr : integer;
begin
   -- write config regs
   for a in 2 to 5 loop
      wait until rising_edge(eclk);
      addr <= conv_std_logic_vector(a, 6);
      din(15 downto 12) <= "0000";
      din(11 downto 9)  <= conv_std_logic_vector(a, 3);
      din(8 downto 0)   <= "111000001";
      wr <= '1';
      cs <= '1';
      wait until rising_edge(eclk);
      wr <= '0';
      cs <= '0';
   end loop;
   
   -- write WC bit, and keep int. period at 1614*4
   wait until rising_edge(eclk);
   addr <= "000001";
   din <= x"9938";
   wr <= '1';
   cs <= '1';
   wait until rising_edge(eclk);
   wr <= '0';
   cs <= '0';
   
   wait for 100 us;
   
   -- set counting pattern test mode
   wait until rising_edge(eclk);
   addr <= "001111";
   din <= x"8000";
   wr <= '1';
   cs <= '1';
   wait until rising_edge(eclk);
   wr <= '0';
   cs <= '0';
   
   -- enable conversions, and keep int. period at 1614*4
   wait until rising_edge(eclk);
   addr <= "000001";
   din <= x"5938";
   wr <= '1';
   cs <= '1';
   wait until rising_edge(eclk);
   wr <= '0';
   cs <= '0';
   
   -- readout for a while
   conv_cntr := 10;
   while conv_cntr > 0 loop
      wait until rising_edge(eclk) and irq = '1';
      readout_cntr := 256;
      while readout_cntr > 0 loop
         addr <= "010000";
         rd <= '1';
         cs <= '1';
         wait until rising_edge(eclk);
         rd <= '0';
         cs <= '0';
         wait until rising_edge(eclk);
         addr <= "010001";
         rd <= '1';
         cs <= '1';
         wait until rising_edge(eclk);
         rd <= '0';
         cs <= '0';
         wait until rising_edge(eclk);
         data <= dout & datal;
         readout_cntr := readout_cntr - 1;
      end loop;
      conv_cntr := conv_cntr - 1;
   end loop;

   -- disable conversions, and keep int. period at 1614*4
   wait until rising_edge(eclk);
   addr <= "000001";
   din <= x"1938";
   wr <= '1';
   cs <= '1';
   wait until rising_edge(eclk);
   wr <= '0';
   cs <= '0';
   
   -- pause for a while
   wait for 500 us;
   
   -- enable conversions, and keep int. period at 1614*4
   wait until rising_edge(eclk);
   addr <= "000001";
   din <= x"5938";
   wr <= '1';
   cs <= '1';
   wait until rising_edge(eclk);
   wr <= '0';
   cs <= '0';
   
   -- readout forever
   loop
      wait until rising_edge(eclk) and irq = '1';
      readout_cntr := 256;
      while readout_cntr > 0 loop
         addr <= "010000";
         rd <= '1';
         cs <= '1';
         wait until rising_edge(eclk);
         rd <= '0';
         cs <= '0';
         wait until rising_edge(eclk);
         addr <= "010001";
         rd <= '1';
         cs <= '1';
         wait until rising_edge(eclk);
         rd <= '0';
         cs <= '0';
         wait until rising_edge(eclk);
         data <= dout & datal;
         readout_cntr := readout_cntr - 1;
      end loop;
   end loop;

   wait; -- halt forever
end process bus_drive;

end sim;
