--- Title: i2c.vhd
---
---
---     o  0                          
---     | /       Copyright (c) 2010  
---    (CL)---o   Critical Link, LLC  
---      \                            
---       O                           
---
--- Company: Critical Link, LLC.
---
--- Revision History:
--- Version  Comments
--- 1.00     Initial Revision
--- 1.01     Add support for 10 bit addressing and interrupt generation

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
library UNISIM;
use UNISIM.VCOMPONENTS.ALL;

library WORK;
use WORK.MityDSP_L138_pkg.ALL;

entity i2c is
   Port ( 
      clk      : in std_logic;
      i_ABus   : in std_logic_vector(5 downto 0);
      i_DBus   : in std_logic_vector(15 downto 0);
      o_DBus   :out std_logic_vector(15 downto 0);
      i_wr_en  : in std_logic;
      i_rd_en  : in std_logic;
      i_cs     : in std_logic;
      o_irq    :out std_logic := '0';
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";      
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";   
      i_sda    : in std_logic;
      o_sda    :out std_logic;
      o_sdt    :out std_logic;
      o_scl    :out std_logic
   );
end i2c;

--*
--* @short Register Transfer Logic Implementation of I2C interface core entity.
--* 
--/
architecture rtl of i2c is

constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 01, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 01, 4);
constant CORE_ID:             std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 07, 8);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 11, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 09, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR( 20, 5);

signal version_reg       : std_logic_vector(15 downto 0);
signal ver_rd            : std_logic;

signal clk_count         : std_logic_vector(15 downto 0) := (others=>'0');
signal clk_div           : std_logic_vector(15 downto 0) := x"0080";
signal clk_out           : std_logic := '0';
signal clk_180, clk_90   : std_logic := '0';

signal ctl_go            : std_logic := '0';
signal ctl_done          : std_logic := '0';
signal ctl_rnw           : std_logic := '0';
signal ctl_word_count    : std_logic_vector(15 downto 0);
signal word_count        : std_logic_vector(15 downto 0);
signal ctl_dev_addr      : std_logic_vector(9 downto 0) := (others=>'0');
signal ctl_addr_10bit    : std_logic := '0';
signal stat_no_ack       : std_logic := '0';
signal ctl_send_stop     : std_logic := '0';
signal low_byte_sent     : std_logic := '0'; -- used when 10 bit addressing mode is active

signal bit_count         : std_logic_vector(3 downto 0) := (others=>'0');
signal data_out          : std_logic_vector(7 downto 0);

signal fifo_in_din       : std_logic_vector(7 downto 0);
signal fifo_in_dout      : std_logic_vector(7 downto 0);
signal fifo_in_wr        : std_logic := '0';
signal fifo_in_rd        : std_logic := '0';
signal fifo_in_full      : std_logic;
signal fifo_in_empty     : std_logic;
signal fifo_in_count     : std_logic_vector(5 downto 0);

signal fifo_out_din       : std_logic_vector(7 downto 0);
signal fifo_out_dout      : std_logic_vector(7 downto 0);
signal fifo_out_wr        : std_logic := '0';
signal fifo_out_rd        : std_logic := '0';
signal fifo_out_full      : std_logic;
signal fifo_out_empty     : std_logic;
signal fifo_out_count     : std_logic_vector(5 downto 0);
signal fifo_rst           : std_logic := '0';

signal done_latch, ctl_done_r1 : std_logic := '0';
signal rx_hf_ie, tx_he_ie, done_ie : std_logic := '0';
signal rx_hf, tx_he : std_logic := '0';

attribute KEEP : string;
signal sda : std_logic;
attribute KEEP of sda : signal is "TRUE";

signal sdt : std_logic := '1';

type I2CSTATES is (IDLE, TX_DATA, RX_DATA, SEND_START, SEND_STOP, CHECK_ACK, SEND_ADDR, GO_IDLE);
signal i2c_state : I2CSTATES := IDLE;

begin

version : core_version
   port map(
      clk           => clk,                  -- system clock
      rd            => ver_rd,               -- read enable
      ID            => CORE_ID,              -- assigned ID number, 0xFF if unassigned
      version_major => CORE_VERSION_MAJOR,   -- major version number 1-15
      version_minor => CORE_VERSION_MINOR,   -- minor version number 0-15
      year          => CORE_YEAR,            -- year since 2000
      month         => CORE_MONTH,           -- month (1-12)
      day           => CORE_DAY,             -- day (1-31)
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg
      );

data2DSP : fifo_dram_sync
    generic map ( 
      WIDTH => 8,
      DEPTH_BITS => 5,
	  RAM_STYLE_HINT => "distributed"
    )
    port map (
			clk     => clk,
			i_din   => fifo_in_din, --From IIC
			i_wr_en => fifo_in_wr,
			i_rd_en => fifo_in_rd,
			o_dout  => fifo_in_dout,  --To DSP
			o_empty => fifo_in_empty,
			o_full  => fifo_in_full,
			o_almost_full => open,
			o_almost_empty => open,
			i_reset => fifo_rst,
			o_count => fifo_in_count
    );

rx_hf <= (fifo_in_count(4) or fifo_in_count(5));
    
data2IIC : fifo_dram_sync
    generic map ( 
      WIDTH => 8,
      DEPTH_BITS => 5,
	  RAM_STYLE_HINT => "distributed"
    )
    port map (
			clk     => clk,
			i_din   => fifo_out_din,
			i_wr_en => fifo_out_wr,
			i_rd_en => fifo_out_rd,
			o_dout  => fifo_out_dout,  --To DSP
			o_empty => fifo_out_empty,
			o_full  => fifo_out_full,
			o_almost_full => open,
			o_almost_empty => open,
			i_reset => fifo_rst,
			o_count => fifo_out_count
    );

tx_he <= not (fifo_out_count(4) or fifo_out_count(5));

reg_read : process (clk)
begin
   if rising_edge(clk) then
      
      ver_rd <= '0';
      fifo_in_rd <='0';

      -- address decoding
      if i_cs = '0' then
         o_DBus <= (others=>'0');
      else
         case i_ABus is
            when "000000" =>
               o_DBus <=  version_reg;
               ver_rd <= i_rd_en;
            when "000001" =>
               o_DBus <= x"000" & '0' & rx_hf_ie & tx_he_ie & done_ie;
            when "000010" =>
               o_DBus <=  x"0" & fifo_out_empty & fifo_in_empty & rx_hf & tx_he & fifo_rst & stat_no_ack & sda & ctl_send_stop & ctl_addr_10bit & ctl_rnw & ctl_go & done_latch;
            when "000011" =>
               o_DBus <= "000000" & ctl_dev_addr;
            when "000100" =>
               o_DBus <=  ctl_word_count; 
            when "000101" =>
               o_DBus <=  clk_div;
            when "000110" =>
               o_DBus <= x"00" & fifo_in_dout;
               fifo_in_rd <= i_rd_en;               
            when others =>
               o_DBus <= (others=>'0');
         end case;
      end if;
   end if;
end process reg_read;

o_irq <= (done_ie and done_latch) or
         (tx_he_ie and tx_he) or
         (rx_hf_ie and rx_hf);

reg_write : process (clk)
begin
   if rising_edge(clk) then

      fifo_out_wr <= '0';
      
      if i2c_state /= IDLE then 
          ctl_go <= '0'; 
      elsif i_cs='1' and i_wr_en='1' and i_ABus ="000010" then
          ctl_go <= i_DBus(1);
      end if;

      ctl_done_r1 <= ctl_done;
      if i_cs='1' and i_wr_en='1' and i_ABus ="000010" then
          if i_DBus(0)='1' then
             done_latch <= '0';
          end if;
      elsif  ctl_done_r1='0' and ctl_done='1' then
          done_latch <= '1';
      end if;

      if i_cs = '1' and i_wr_en = '1' then
         case i_ABus is
            when "000001" =>
                rx_hf_ie <= i_DBus(2);
                tx_he_ie <= i_DBus(1);
                done_ie  <= i_DBus(0);
            when "000010" =>
                ctl_rnw <= i_DBus(2);
                ctl_addr_10bit <= i_DBus(3);
                ctl_send_stop <= i_DBus(4);
                fifo_rst <= i_DBus(7);
            when "000011" =>	
                ctl_dev_addr <= i_DBus(9 downto 0);	
            when "000100" =>
                ctl_word_count <= i_DBus;
            when "000101" =>
                clk_div <= i_DBus;
            when "000110" =>
                fifo_out_wr <= '1';
                fifo_out_din <= i_DBus(7 downto 0);
            when others => NULL;
         end case;
      end if;
   end if;
end process reg_write;

clk_divide : process(clk)
begin
    if rising_edge(clk) then
        if clk_count >= clk_div then
           clk_out <= not clk_out;
           clk_count <= (others=>'0');
           clk_180 <= '1';
        else
           if clk_count = '0' & clk_div(15 downto 1) then
              clk_90 <= '1';
           else 
              clk_90 <= '0';
           end if;
           clk_180 <= '0';
           clk_count <= clk_count + '1';
        end if;
    end if;
end process clk_divide;

o_scl <= clk_out when (i2c_state /= IDLE and i2c_state /= SEND_START) else '1';
o_sdt <= sdt;
o_sda <= '0';

i2c_sm : process(clk)
begin
   if rising_edge(clk) then
   
      fifo_in_wr <= '0';
      fifo_out_rd <= '0';
      sda <= i_sda;
      
      case i2c_state is
         when IDLE =>
            sdt <= '1';
            if ctl_go = '1' then
               ctl_done <= '0';
               i2c_state <= SEND_START;
               word_count <= ctl_word_count;
               stat_no_ack <= '0';
               low_byte_sent <= '0';
            end if;
            
         when SEND_START =>
            -- ceneter of high clock period
            if clk_90='1' and clk_out='1' then
               sdt <= '0';
               i2c_state  <= SEND_ADDR;
               if ctl_addr_10bit = '1' then -- not quite done yet
                  data_out   <= "11110" & ctl_dev_addr(9 downto 8) & ctl_rnw; 
               else
                  data_out   <= ctl_dev_addr(6 downto 0) & ctl_rnw; 
               end if;
               bit_count  <= (others=>'0');
            end if;
            
         when SEND_ADDR =>
            -- center of low clock period
            if clk_90='1' and clk_out='0' then
               data_out <= data_out(6 downto 0) & '0';
               if bit_count = conv_std_logic_vector(8, 4) then
                  i2c_state <= CHECK_ACK;
                  sdt <= '1'; -- tristate to set up for ack check
               else
                  sdt <= data_out(7);
                  bit_count <= bit_count+'1';
               end if;
            end if;
            
         when CHECK_ACK =>
             -- check on center of high clock state
             if clk_90='1' and clk_out='1' then
                if sda ='0' then
                    -- if we need to send the remainder of a 10 bit slave address
                    if low_byte_sent='0' and ctl_addr_10bit = '1' then
                        sdt <= '0';
                        i2c_state  <= SEND_ADDR;
                        data_out   <= ctl_dev_addr(7 downto 0); 
                        bit_count  <= (others=>'0');
                        low_byte_sent <= '1';
                    -- else it's time for the data shuffle
                    elsif word_count /= conv_std_logic_vector(0, 16) then
                        bit_count  <= (others=>'0');
                        if ctl_rnw='1' then
                           i2c_state <= RX_DATA;
                        else
                           data_out <= fifo_out_dout;
                           i2c_state <= TX_DATA;
                        end if;
                    elsif ctl_send_stop='1' then
                        i2c_state <= SEND_STOP;
                    else
                        i2c_state <= GO_IDLE;
                    end if;
                else -- error condition, not ack'd
                    -- on the last read, we assert a NAK to tell other end we're done
                     -- this is not an error
                    if ctl_rnw='0' or word_count /= conv_std_logic_vector(0, 16) then
                       stat_no_ack <= '1';
                    end if;
                    i2c_state <= SEND_STOP;
                end if;
             end if;
             
         when GO_IDLE =>
             -- tristate output at falling edge
             if clk_90='1' and clk_out='0' then
                 sdt <= '1';
             end if;
             
             -- finish off at clock rising edge
             if clk_180='1' and clk_out='1' then
                 i2c_state <= IDLE;
                 ctl_done <= '1';
             end if;
             
         when RX_DATA =>
           -- make sure tristate logic is set (falling edge of clock)
           if clk_180='1' and clk_out='0' then
                  sdt <= '1';
           end if;
           
           -- read on center of high clock state
           if clk_90='1' and clk_out='1' then
              fifo_in_din <= fifo_in_din(6 downto 0) & sda;
              bit_count <= bit_count+'1';
              if bit_count = conv_std_logic_vector(7, 4) then
                 fifo_in_wr <= '1';
                 word_count <= word_count-'1';
              end if;
           end if;

           -- when done reading, drive SDA low (we need to assert an ACK when reading)
           if clk_90='1' and clk_out='0' and bit_count = conv_std_logic_vector(8, 4) then
               i2c_state <= CHECK_ACK; 
               if word_count /= conv_std_logic_vector(0,16) then
                   sdt <= '0';
               else
                   sdt <= '1';
               end if;
           end if;
         
         when TX_DATA =>
           -- update output on center of low edge
           if clk_90='1' and clk_out='0' then
              data_out <= data_out(6 downto 0) & '0';
              if bit_count = conv_std_logic_vector(8,4) then
                 fifo_out_rd <= '1';
                 word_count <= word_count-'1';
                 i2c_state <= CHECK_ACK;
                 sdt <='1';
              else
                 sdt <= data_out(7);
                 bit_count <= bit_count+'1';
              end if;
           end if;
         
         when SEND_STOP =>
           -- send a stop condition
           if clk_90='1' and clk_out='0' then
              sdt <= '0';
           end if;
           if clk_90='1' and clk_out='1' then
              sdt <= '1';
              i2c_state <= IDLE;
              ctl_done <= '1';
           end if;
         
         when others => NULL;
      end case;
   end if;
end process i2c_sm;

end rtl;
