--- Title: lcd_ctlr.vhd
--- Description: 
---
---     o  0
---     | /       Copyright (c) 2010
---    (CL)---o   Critical Link, LLC
---      \
---       O
---
--- Company: Critical Link, LLC.
--- Date: 3/7/2010
--- Version: 1.00
--- Revisions: 
---   1.00 - Initial release.
---   1.01 - Fix PWM to support 0 percent duty cycle.
---        - add lcd reset logic

library IEEE;
library UNISIM;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use UNISIM.VCOMPONENTS.ALL;

library WORK;
use WORK.MityDSP_L138_pkg.ALL;

entity lcd_ctlr is
   port (
      ema_clk         : in std_logic;
      i_cs            : in std_logic;
      i_ABus          : in std_logic_vector(5 downto 0);
      i_DBus          : in std_logic_vector(15 downto 0);
      o_DBus          : out std_logic_vector(15 downto 0);
      i_wr_en         : in std_logic;
      i_rd_en         : in std_logic;
      o_irq           : out std_logic;
      i_ilevel       : in std_logic_vector(1 downto 0) := "00";       -- interrupt level (0=4,1=5,2=6,3=7)
      i_ivector      : in std_logic_vector(3 downto 0) := "0000";    -- interrupt vector (0 through 31)
      
      o_lcd_hrev    : out std_logic;
      o_lcd_dim_pwm : out std_logic; -- high is a "dim" or turn off state
      o_lcd_pwr     : out std_logic;
      o_lcd_backlit : out std_logic;
      o_lcd_vrev    : out std_logic;
      o_lcd_en      : out std_logic;
      o_lcd_reset_stb : out std_logic -- reset output (active high)
                  
      );
end lcd_ctlr;

architecture rtl of lcd_ctlr is

constant CORE_ID:             std_logic_vector(7 downto 0) := CONV_STD_LOGIC_VECTOR( 2, 8);
constant CORE_VERSION_MAJOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 1, 4);
constant CORE_VERSION_MINOR:  std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR( 0, 4);
constant CORE_YEAR:           std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR(10, 5);
constant CORE_MONTH:          std_logic_vector(3 downto 0) := CONV_STD_LOGIC_VECTOR(03, 4);
constant CORE_DAY:            std_logic_vector(4 downto 0) := CONV_STD_LOGIC_VECTOR(08, 5);

signal version_reg : std_logic_vector(15 downto 0) := (others=>'0');
signal version_rd  : std_logic;

signal lcd_hrev    : std_logic := '1';
signal lcd_dim_pwm : std_logic := '0';
signal lcd_pwr     : std_logic := '1';
signal lcd_backlit : std_logic := '1';
signal lcd_vrev    : std_logic := '0';
signal lcd_en      : std_logic := '0';
signal lcd_reset   : std_logic := '0';
signal pwm_clocks  : std_logic_vector(15 downto 0) := x"3000";
signal pwm_on_count : std_logic_vector(15 downto 0) := x"0000";
signal pwm_count    : std_logic_vector(15 downto 0) := x"0000";

begin

o_irq <= '0';

version : core_version
   port map(
      clk           => ema_clk,
      rd            => version_rd,
      ID            => CORE_ID,
      version_major => CORE_VERSION_MAJOR,
      version_minor => CORE_VERSION_MINOR,
      year          => CORE_YEAR,
      month         => CORE_MONTH,
      day           => CORE_DAY,
      ilevel        => i_ilevel,
      ivector       => i_ivector,
      o_data        => version_reg
      );

reg_read : process (ema_clk)
begin
   if rising_edge(ema_clk) then
      
      version_rd <= '0';
      
      if i_cs='0' then
         o_DBus <= (others=>'0');
      else
         case i_ABus is
            when "000000" =>
               o_DBus <= version_reg;
               version_rd <= i_rd_en;
            when "000010" =>
               o_DBus <= x"00" & "00" & lcd_reset & lcd_vrev & lcd_hrev & lcd_backlit & lcd_pwr & lcd_en;
            when "000011" =>
               o_DBus <= pwm_clocks;
            when "000100" =>
               o_DBus <= pwm_on_count;
            when others =>
               o_DBus <= (others=>'0');
         end case;
      end if;
   end if;
end process reg_read;

reg_write : process (ema_clk)
begin
   if rising_edge(ema_clk) then
      if i_cs='1' and i_wr_en='1' then
               
         case i_ABus is
            when "000010" => NULL;
               lcd_en      <= i_DBus(0);
               lcd_pwr     <= i_DBus(1);
               lcd_backlit <= i_DBus(2);
               lcd_hrev    <= i_DBus(3);
               lcd_vrev    <= i_DBus(4);
               lcd_reset   <= i_DBus(5);
            when "000011" =>
               pwm_clocks <= i_DBus;
            when "000100" =>
               pwm_on_count <= i_DBus;
            when others => NULL;
         end case;
         
      end if;
   end if;
end process reg_write;

pwm : process (ema_clk)
begin
   if rising_edge(ema_clk) then
       if pwm_count > pwm_clocks then
         pwm_count <= (others=>'0');
       else
         pwm_count <= pwm_count+'1';
       end if;
       if pwm_count < pwm_on_count then
         lcd_dim_pwm <= '0';
       else
         lcd_dim_pwm <= '1';
       end if;
   end if;
end process pwm;

o_lcd_hrev    <= lcd_hrev;
o_lcd_dim_pwm <= lcd_dim_pwm;
o_lcd_pwr     <= lcd_pwr;
o_lcd_backlit <= lcd_backlit;
o_lcd_vrev    <= lcd_vrev;
o_lcd_en      <= lcd_en;
o_lcd_reset_stb <= lcd_reset;

end rtl;
