/**
 * \file  ipc.c
 *
 * \brief Top level APIs for lightweight IPC
 */

/* Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/* general IPC header files */
#include "ipc.h"
#include "notify.h"

/* internal IPC header files */
#include "ipc_pv.h"

extern int  ipc_NotifySetup(struct IPC_cfg *, struct IPC_modControl *);
extern void ipc_ClearIpcInt(void);
extern int  ipc_RegisterInt(int);
extern void ipc_NotifyIntInfo (int);
extern int  ipc_NotifySyncRemote (void);
extern int  ipc_InitSyncRemote (void);

static struct IPC_modControl IpcInstObj; /* Create one IPC control instance */

/******************************************************************************
**                  IPC Module API Functions
******************************************************************************/

/* IpcInit */
Int IPC_init(struct IPC_cfg *IpcObj)
{
    int status = Ipc_E_FAIL;
    /* Check system key to see if this is cold boot */
    if ( IpcObj->localEvtPtr->key == IPC_BOOT_KEY)
    {
        /* Abort if already in ready or running state */
        if ( (IpcInstObj.ipcState == IPC_STATE_READY)   ||
             (IpcInstObj.ipcState == IPC_STATE_RUNNING) ||
             (IpcInstObj.ipcState == IPC_STATE_RUNNINGPOLLEDINT) )
        status = Ipc_S_ALREADYSETUP;
    }
    else
    {
        IpcInstObj.ipcState         =     IPC_STATE_NOTREADY;
        IpcObj->localEvtPtr->key     =     IPC_BOOT_KEY;
        IpcObj->localEvtPtr->semCnt =    0;
        IpcInstObj.remoteProcId     =    IpcObj->remoteProcId ;
        IpcInstObj.remoteIntLine    =    IpcObj->remoteIntLine;
    
        /* Create Notify control & data structures */
        status = ipc_NotifySetup(IpcObj, &IpcInstObj);

        if (status == Notify_S_SUCCESS)
        {
            IpcInstObj.ipcState = IPC_STATE_READY;
            /* Wait for remote processor to start up IPC_init */
            ipc_InitSyncRemote();
            status = Ipc_S_SUCCESS;
        }
    }
    return status;
}


int IPC_intRegister(unsigned int val)
{
    ipc_ClearIpcInt();
    ipc_NotifyIntInfo (val);
    return (ipc_RegisterInt(val));
}

/******************************************************************************
**                  Internal Functions
******************************************************************************/


/* END OF FILE */
