/**
 * \file  main_dsp.c
 *
 * \brief Simple IPC example using polling method (DSP side)
 */

/* Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

#include "ipc_polling.h"

/* Globals */
int    volatile evtRcvd;
UInt32 volatile myPayload;

/* Declarations */
void AppInit(void);
void iAssert (int);

/* Event Call back function */
void cbFxnEvent(UInt16 procId, UInt16 lineId,
                UInt32 eventId, UArg arg, UInt32 payload)
{
    myPayload = payload; /* Transfer payload */
    evtRcvd++;           /* Flag event to background app */
}

void main(void)
{
    int intCnt = 0;
    /* Initialization */
    evtRcvd = 0;
    AppInit();
    printf ("DSP IPC ready\n");

    /* Send an event */
    myPayload = 0x1;
    iAssert (Notify_sendEvent(IPC_ARM_HOST,IPC_HOST_INTLINE, EVENTID, myPayload, TRUE));

    /* Wait for event from DSP */
    while (intCnt < IPC_LOOP_COUNT)
    {
        iAssert (Notify_waitEvent(IPC_ARM_HOST,IPC_HOST_INTLINE));
        intCnt++;
        printf("Event received %x\n", myPayload);
        iAssert (Notify_sendEvent(IPC_ARM_HOST,IPC_HOST_INTLINE, EVENTID, myPayload << 1, FALSE));
    }

    /* Done - exit app */
    printf("dsp: done - total event received %d\n", intCnt);
    WAIT_HERE;
}


/* Initialization */
void AppInit(void)
{
    /* Structure to initialize IPC (see Ipc.h for definitions) */
    struct IPC_cfg ipcCfg = {
            IPC_DSP0,           /* host processor Id */
            IPC_ARM_HOST,       /* remote processor Id */
            IPC_HOST_INTLINE,   /* ID of interrupt line to remote CPU */
            IPC_POLLING_METHOD, /* Method to receive from remote processor */
            NUM_MAX_EVENT,      /* maximum number of events to be created */
            &ipcPvMemDsp,       /* local side private IPC memory */
            &ipcPvMemArm        /* remote side private IPC memory */
    };
    IntDSPINTCInit();
    IPC_init(&ipcCfg);
    IPC_intRegister(C674X_MASK_INT5);
    iAssert (Notify_registerEvent(IPC_ARM_HOST, IPC_HOST_INTLINE, EVENTID, cbFxnEvent, 0xc0de));

    /*  Enabling interrupts for DSP C674x CPU */
    IntGlobalEnable();        /* Enable C674x global interrupt */
    iAssert (Notify_start()); /* This will enable IPC interrupt */
}

/* Trap for errors */
void iAssert (int status)
{
    if (status != Notify_S_SUCCESS)
    {
        printf("Notify failed %d\n", status);
        WAIT_HERE;
    }
}
