/**
 * \file  notify_driver_OMAPL138.c
 *
 * \brief Low-level hardware support routines for IPC Notify module for
 *        OMAP-L138
 */

/* Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

#include "hw_types.h"

#ifdef _TMS320C6X
#include "soc_C6748.h"
#include "hw_syscfg0_C6748.h"
#include "interrupt.h"
#else
#include "soc_AM1808.h"
#include "hw_syscfg0_AM1808.h"
#include "cpu.h"
#include "interrupt.h"
#endif
 
 /* internal IPC header files */
#include "notify_pv.h"

/******************************************************************************
**                  IPC Function Definitions
******************************************************************************/
void    ipc_ClearIpcInt(void);
void    ipc_SetIpcInt(void);
int        ipc_CheckIpcIntStatus (void);

extern    void    ipc_NotifyIsrAckEvent(void);

/******************************************************************************
**                  IPC Notify ISRs
******************************************************************************/

/* DSP0 IPC Interrupt Service Routine */
#ifdef _TMS320C6X
void    ipc_NotifyIsr(void)
{
    IntDisable (C674X_MASK_INT5);
    /* ISR work starts */
    ipc_NotifyIsrAckEvent();
    /* Clear interrupt status and re-enable interrupt */
    ipc_ClearIpcInt();
    IntEnable (C674X_MASK_INT5);
}
/* Register interrupt for DSP INTC */
int ipc_RegisterInt(unsigned int val)
{
    IntRegister(val, ipc_NotifyIsr);           /* Register the ipc ISR */
    IntEventMap(val, SYS_INT_SYSCFG_CHIPINT2); /* Map this event to this interrupt */
    return (Ipc_S_SUCCESS);
}
/* Enable IPC interrupt for local CPU */
int ipc_NotifyIntEnable(void)
{
    IntEnable (C674X_MASK_INT5);
    return (Ipc_S_SUCCESS);
}
/* Disable IPC interrupt for local CPU */
int ipc_NotifyIntDisable(void)
{
    IntDisable (C674X_MASK_INT5);
    return (Ipc_S_SUCCESS);
}
/* Clear interrupt to DSP CPU
 * OMAP-L138: use SYSCFG0 CHIPINT2 */
void ipc_ClearIpcInt(void)
{
     unsigned int temp;
     temp = HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG_CLR) & SYSCFG_CHIPSIG_CHIPSIG2;
     temp |= SYSCFG_CHIPSIG_CHIPSIG2;
     HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG_CLR) = temp;
     IntEventClear(SYS_INT_SYSCFG_CHIPINT2);
}
/* Set interrupt to Host (ARM) CPU
 * OMAP-L138: use SYSCFG0 CHIPINT0 */
void ipc_SetIpcInt(void)
{
     unsigned int temp;
     /* set the SYSCFG0 CHIPINT0 bit to interrupt ARM9 (OMAP-L138) */
     temp = HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) & SYSCFG_CHIPSIG_CHIPSIG0;
     temp |= SYSCFG_CHIPSIG_CHIPSIG0;
     HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) = temp;
}

/* Check status of IPC interrupt to DSP0
 * OMAP-L138:    C674x CPU */
int        ipc_CheckIntStatus(void)
{
    return (HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) & SYSCFG_CHIPSIG_CHIPSIG2);
}
/* Check using SYSCFG0 CHIPINT instead */
int        ipc_CheckIpcIntStatus (void)
{
    return (HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) & SYSCFG_CHIPSIG_CHIPSIG2);
}
#else
/* Host IPC Interrupt Service Routine */
void    ipc_NotifyIsr(void)
{
    IntSystemDisable(SYS_INT_SYSCFG_CHIPINT0);
    /* ISR work starts
     * Check to make sure IPC int is received instead of spurious */
    if     ((HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) & SYSCFG_CHIPSIG_CHIPSIG0)== SYSCFG_CHIPSIG_CHIPSIG0)
    {
        ipc_NotifyIsrAckEvent();
        /* Clear interrupt status and re-enable interrupt */
        ipc_ClearIpcInt();
    }
    IntSystemEnable(SYS_INT_SYSCFG_CHIPINT0);
}

int ipc_RegisterInt(unsigned int val)
{
    IntRegister(SYS_INT_SYSCFG_CHIPINT0, ipc_NotifyIsr); /* register ipc ISR */
    IntChannelSet(SYS_INT_SYSCFG_CHIPINT0, val);         /* select channel   */
    return Ipc_S_SUCCESS;
}
/* Enable IPC interrupt for local CPU */
int ipc_NotifyIntEnable(void)
{
    IntSystemEnable(SYS_INT_SYSCFG_CHIPINT0); /* Enable ipc interrupts in AINTC */
    return Ipc_S_SUCCESS;
}
/* Disable IPC interrupt for local CPU */
int ipc_NotifyIntDisable(void)
{
    IntSystemDisable(SYS_INT_SYSCFG_CHIPINT0);
    return Ipc_S_SUCCESS;
}
/* Clear IPC interrupt flag to ARM CPU
 * OMAP-L138: use SYSCFG0 CHIPINT0 */
void ipc_ClearIpcInt(void)
{
        unsigned int temp;
         /* Clear SYSCFG0 CHIPINT0 interrupt flag to ARM */
         temp = HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG_CLR) & SYSCFG_CHIPSIG_CHIPSIG0;
         temp |= SYSCFG_CHIPSIG_CHIPSIG0;
         HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG_CLR) = temp;
         IntSystemStatusClear(SYS_INT_SYSCFG_CHIPINT0);
}

/* Set interrupt to DSP CPU
 * OMAP-L138: use SYSCFG0 CHIPINT2 */
void    ipc_SetIpcInt(void)
{
    unsigned int temp;
     /* Set SYSCFG0 CHIPINT0 interrupt flag to DSP */
     temp = HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) & SYSCFG_CHIPSIG_CHIPSIG2;
     temp |= SYSCFG_CHIPSIG_CHIPSIG2;
     HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) = temp;
}

/* Check status of IPC interrupt to Host */
int ipc_CheckIntStatus(void)
{
    return HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) & SYSCFG_CHIPSIG_CHIPSIG0;
}

/* Check using SYSCFG0 CHIPINT instead */
int ipc_CheckIpcIntStatus (void)
{
    return HWREG(SOC_SYSCFG_0_REGS + SYSCFG0_CHIPSIG) & SYSCFG_CHIPSIG_CHIPSIG0;
}
#endif

/*    end notify_driver_OMAPL138.c    */
