/**
 * \file  notify.c
 *
 * \brief Notification module for IPC
 */

/* Copyright (c) 2011, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */

/*  A Note about interrupt latency:
 *
 *  Some of the functions below modify the registration of callback functions
 *  (registerEvent, unregisterEvent, registerEventMany, unregisterEventMany)
 *  Because the state of event registration is modified in these functions and
 *  the Notify_exec function (run within an ISR context) reads this state, we
 *  have to consider the possibility and outcome of these functions being
 *  interrupted by the Notify_exec function.  In order to ensure the integrity
 *  of the callback registration state during any call to Notify_exec, we
 *  eliminate the possibility of preemption altogether by adhering to the
 *  following rules:
 *
 *  - If registering a single (or many) callback functions, modify state in the
 *    following order:
 *
 *    1.  Add to the event list (if registerMany)
 *    2.  Add the callback function (if registerMany for the first time or
 *        if doing a single register)
 *    3.  Finally enable the event in the notify driver, thus opening the gate
 *        for incoming interrupts.
 *
 *    Performing step 3) last during a register is crucial as it ensures that
 *    the remote processor never sends an interrupt that disrupts the operations
 *    performed in/between 1) and 2).
 *
 *  - If unregistering a single (or many) callback functions, perform the
 *    above steps in reverse order.  Doing so ensures that possibility of
 *    getting incoming interrupts is eliminated before local callback
 *    registration state is modified. */

#include <stdlib.h>

/* general IPC header files */
#include "ipc.h"
#include "notify.h"

/* internal IPC header files */
#include "ipc_pv.h"
#include "notify_pv.h"

/* External */
extern int  ipc_NotifyIntEnable(void);
extern int  ipc_NotifyIntDisable(void);
extern void ipc_SetIpcInt(void);
extern void ipc_ClearIpcInt(void);
extern int  ipc_CheckIntStatus(void);

/* Internal */
void ipc_NotifyIsrAckEvent(void);
int  ipc_NotifySyncRemote (void);

/* Create private Notify object */
static struct IPC_notifyCfg ntfyCfgObj;
/* Event slot control structure */
static IPC_notifyEventSlot esTbl[Notify_MAXEVENTS];

/******************************************************************************
**                  Notify Module-wide API Functions
******************************************************************************/

/**
 * \brief  Start Notify mechanism by synchronizing with remote processor.
 *         IPC Interrupt will be enabled if IPC_INTERRUPT_METHOD was selected
 *
 * \return Notify status:
 *         - #Notify_E_FAIL: failed to connect to remote processor
 *         - #Notify_S_SUCCESS: successfully start
 */
Int    Notify_start(void)
{
    int    status;

    /* Clear any pending IPC interrupt since we are starting
     * Sync the processors and change state on successful connect */
    ipc_ClearIpcInt();
    status = ipc_NotifySyncRemote();
    if (status == Notify_S_SUCCESS)
    {
        if (ntfyCfgObj.NotifyRcvMethod == IPC_INTERRUPT_METHOD)
        {
            ipc_NotifyIntEnable();
            ntfyCfgObj.IpcObjHandle->ipcState = IPC_STATE_RUNNING;
        }
        else
            ntfyCfgObj.IpcObjHandle->ipcState = IPC_STATE_RUNNINGPOLLEDINT;
    }
    return status;
}


/**
 * \brief     Stop receiving notification from remote CPU via interrupts.
 *
 * \param[in] procId               Remote processor id

 * \return    Notify status:
 *            - #Notify_E_INVALIDARG: invalid procId or lineId value
 *            - #Notify_E_INVALIDSTATE: Notify disable failed - IPC is not in running state
 *            - #Notify_E_FAIL: Notify disable failed - IPC already in polling state or
 *              IPC_POLLING_METHOD was selected in configuration.
 *            - #Notify_S_SUCCESS: IPC Notification received via interrupt stopped
 */
Int Notify_disable(UInt16 procId, UInt16 lineId)
{
    /* procId and lineId are not used in this version but checked here */
    int status = Notify_E_INVALIDSTATE;
    if ( ( procId != ntfyCfgObj.eventTblHandle[0].remoteProcId )    ||
         ( lineId != ntfyCfgObj.eventTblHandle[0].remoteIntLine )    )
        status = Notify_E_INVALIDARG;
    /* Proceed only if the current IPC state is running */
    else if ( ntfyCfgObj.IpcObjHandle->ipcState == IPC_STATE_RUNNING )
    {
            if (ntfyCfgObj.NotifyRcvMethod == IPC_INTERRUPT_METHOD)
            {
                ipc_NotifyIntDisable();
                ntfyCfgObj.IpcObjHandle->ipcState = IPC_STATE_RUNNINGPOLLEDINT;
                status = Notify_S_SUCCESS;
            }
            else
                status = Notify_E_FAIL;
    }

    return (status);
}


/**
 * \brief      Restore Notify receive event notification from
 *             remote CPU via interrupt
 *
 * \param[in]  procId               Remote processor id
 * \params[in] lineId               " interrupt line id
 *
 * \return     Notify status:
 *             - #Notify_E_INVALIDARG: invalid procId or lineId value
 *             - #Notify_E_INVALIDSTATE: Notify disable failed - IPC is not in polling state
 *             - #Notify_E_FAIL: Notify disable failed - IPC already in running state or
 *               IPC_POLLING_METHOD was selected in configuration.
 *             - #Notify_S_SUCCESS: successfully start
 */
Int Notify_restore(UInt16 procId, UInt16 lineId)
{
    /* procId and lineId are not used in this version but checked here */
    int status = Notify_E_INVALIDSTATE;
    if ( ( procId != ntfyCfgObj.eventTblHandle[0].remoteProcId )    ||
         ( lineId != ntfyCfgObj.eventTblHandle[0].remoteIntLine )    )
        status = Notify_E_INVALIDARG;

    /* Proceed only if the current IPC state is running in polled int mode */
    else if ( ntfyCfgObj.IpcObjHandle->ipcState == IPC_STATE_RUNNINGPOLLEDINT )
    {
        if (ntfyCfgObj.NotifyRcvMethod == IPC_INTERRUPT_METHOD)
        {
            status = ipc_NotifyIntEnable();
            ntfyCfgObj.IpcObjHandle->ipcState = IPC_STATE_RUNNING;
            status = Notify_S_SUCCESS;
        }
        else
            status = Notify_E_FAIL;
    }
    return status;
}
/**
 * \brief      Wait for any event
 *
 * \params[in] rmCpuId        from this Remote processor id
 * \param[in]  lnId          on this interrupt Line id
 * 
 * \return     Notify status:
 *             - #Notify_E_FAIL: cannot perform action - event not from remote CPU ID
 *             - #Notify_E_EVTNOTREGISTERED: event has no registered callback function
 *             - #Notify_S_SUCCESS: successfully received event from remote processor
 */
Int Notify_waitEvent(UInt16 rmCpuId, UInt16 lnId)
{
    int status = Notify_E_FAIL;

    /* Wait for IPC event */
    int temp;
    do temp = ipc_CheckIntStatus();
    while (!temp);

    /* This API is allowed only if the event is registered and IPC
     * is in polling mode. */
    if ( (rmCpuId == ntfyCfgObj.IpcObjHandle->remoteProcId)    &&
         (lnId == ntfyCfgObj.IpcObjHandle->remoteIntLine)        )
    {
        ipc_NotifyIsrAckEvent();
        status = (int) ntfyCfgObj.localEventPtr->payload;
        ipc_ClearIpcInt();
    }
    return status;
}

/**
 * \brief     Disable an event
 *
 * \param[in] procId      Remote processor id
 * \param[in] lineId      Line id
 * \param[in] eventId     Event id
 *
 * \sa        Notify_enableEvent
 */
Void Notify_disableEvent(UInt16 procId, UInt16 lineId, UInt32 eventId)
{
    IPC_notifyEventSlot *pEvtSlot = ntfyCfgObj.eventTblHandle + eventId;
    /* Change state only if the event slot was registerd otherwise ignore */
    if (pEvtSlot->status != Notify_EVENT_SLOT_FREE)
        pEvtSlot->status = Notify_EVENT_SLOT_DISABLED;
}

/*!
 * \brief      Enable an event
 *
 * \param[in]  procId      Remote processor id
 * \param[in]  lineId      Line id
 * \param[in]  eventId     Event id
 *
 * \sa         Notify_disableEvent
 */
Void Notify_enableEvent(UInt16 procId, UInt16 lineId, UInt32 eventId)
{
    IPC_notifyEventSlot *pEvtSlot = ntfyCfgObj.eventTblHandle + eventId;
    /* Change state only if the event slot was disabled otherwise ignore */
    if (pEvtSlot->status == Notify_EVENT_SLOT_DISABLED)
        pEvtSlot->status = Notify_EVENT_SLOT_REGISTED;
}

/*!
 * \brief      Register a callback for an event
 *
 * \param[in]  procId          Remote processor id
 * \param[in]  lineId          Line id (0 for most systems)
 * \param[in]  eventId         Event id
 * \param[in]  fnNotifyCbck    Pointer to callback function
 * \param[in]  cbckArg         Callback function argument
 *
 * \return     Notify status:
 *              - #Notify_E_INVALIDARG: invalid procId or lineId value
 *              - #Notify_E_FAIL: cannot perform action - event already registered
 *              - #Notify_S_SUCCESS: Event successfully registered
 *
 * \sa         Notify_unregisterEvent
 */
Int Notify_registerEvent(UInt16 procId, 
                         UInt16 lineId, 
                         UInt32 eventId,
                         Notify_FnNotifyCbck fnNotifyCbck,
                         UArg cbckArg)
{
    int status = Notify_E_FAIL;
    if ( ( procId != ntfyCfgObj.eventTblHandle[eventId].remoteProcId )    ||
         ( lineId != ntfyCfgObj.eventTblHandle[eventId].remoteIntLine )    )
        status = Notify_E_INVALIDARG;
    else if (ntfyCfgObj.eventTblHandle[eventId].status == Notify_EVENT_SLOT_FREE)
    {
        ntfyCfgObj.eventTblHandle[eventId].cbFxn = fnNotifyCbck;
        ntfyCfgObj.eventTblHandle[eventId].status = Notify_EVENT_SLOT_REGISTED;
        ntfyCfgObj.eventTblHandle[eventId].cbKey = cbckArg;
        /* Check if events are not over registered */
        if (ntfyCfgObj.numRegistedEvents < ntfyCfgObj.maxNumofEvents)
        {
                ntfyCfgObj.numRegistedEvents++;
                status = Notify_S_SUCCESS;
        }
        else
        {
            status = Notify_E_INVALIDARG; /* this is an error condition */
        }
    }
    return (status);
}

/*!
 * \brief      Unregister a callback for an event (supports multiple callbacks)
 *
 * \param[in]  procId          Remote processor id
 * \param[in]  lineId          Line id (0 for most systems)
 * \param[in]  eventId         Event id
 * \param[in]  fnNotifyCbck    Pointer to callback function
 * \param[in]  cbckArg         Callback function argument
 *
 * \return     Notify status:
 *              - #Notify_E_NOTFOUND: event listener (callback) not found
 *              - #Notify_E_INVALIDARG: invalid procId or lineId value
 *              - #Notify_E_FAIL: cannot perform action - event slot free (unregistered) or has data
 *              - #Notify_S_SUCCESS: Event successfully registered
 *
 * \sa         Notify_unregisterEvent
 */
Int Notify_unregisterEvent(UInt16 procId,
                         UInt16 lineId,
                         UInt32 eventId,
                         Notify_FnNotifyCbck fnNotifyCbck,
                         UArg cbckArg)
{
    int status = Notify_E_FAIL;
    if ( ( procId != ntfyCfgObj.eventTblHandle[eventId].remoteProcId )    ||
         ( lineId != ntfyCfgObj.eventTblHandle[eventId].remoteIntLine )    )
        status = Notify_E_INVALIDARG;
    else if (ntfyCfgObj.eventTblHandle[eventId].cbFxn != fnNotifyCbck)
        status = Notify_E_NOTFOUND;
    else if (ntfyCfgObj.eventTblHandle[eventId].status == Notify_EVENT_SLOT_REGISTED)
    {
        ntfyCfgObj.eventTblHandle[eventId].cbFxn = fnNotifyCbck;
        ntfyCfgObj.eventTblHandle[eventId].status = Notify_EVENT_SLOT_FREE;
        if (ntfyCfgObj.numRegistedEvents > 0)
        {
                ntfyCfgObj.numRegistedEvents--;
                status = Notify_S_SUCCESS;
        }
        else
        {
            /* this is an error condition */
            ntfyCfgObj.numRegistedEvents = 0;
            status = Notify_E_INVALIDARG;
        }
    }
    return (status);
}

/*!
 * \brief      Send an event on an interrupt line
 *
 *  Notify_sendEvent can be called from a Hwi context unlike other Notify APIs.
 *
 * \param[in]  procId      Remote processor id
 * \param[in]  lineId      Line id
 * \param[in]  eventId     Event id
 * \param[in]  payload     Payload to be sent along with the event.
 * \param[in]  waitClear   Indicates whether Notify driver will wait for
 *                          previous event to be cleared.
 *  
 * \return     Notify status:
 *              - #Notify_E_INVALIDARG: invalid procId or lineId value
 *              - #Notify_E_EVTNOTREGISTERED: event has no registered callback 
 *                functions
 *              - #Notify_E_NOTINITIALIZED: remote driver has not yet been
 *                initialized
 *              - #Notify_E_EVTDISABLED: remote event is disabled
 *              - #Notify_E_BUSY: unprocessed event with this eventId on remote side
 *              - #Notify_E_TIMEOUT: timeout occured (when waitClear is TRUE)
 *              - #Notify_S_SUCCESS: event successfully sent
 */
Int Notify_sendEvent(UInt16 procId, 
                     UInt16 lineId,
                     UInt32 eventId, 
                     UInt32 payload,
                     Bool waitClear)
{
    int    status;
    IPC_notifyEvent *evtPtr = ntfyCfgObj.remoteEventPtr;

    /* Process arguments */
    if ((eventId > ntfyCfgObj.maxNumofEvents)                 ||
        (procId != ntfyCfgObj.IpcObjHandle->remoteProcId)    ||
        (lineId != ntfyCfgObj.IpcObjHandle->remoteIntLine)    )
        status = Notify_E_INVALIDARG;
    else
    {
        /* Transfer args to ipc private area */
        evtPtr->procId         =     procId;
        evtPtr->lineId         =    lineId;
        evtPtr->eventId     =     eventId;
        evtPtr->payload     =     payload;
        evtPtr->semCnt == 0 ? (evtPtr->semCnt++) : (evtPtr->semCnt=0);
        /* Send event to remote processor */
        status = ntfyCfgObj.ipcNotifyFxn(waitClear);
    }
    return    status;
}                       

/******************************************************************************
**                  Notify Module Internal Functions
******************************************************************************/
/* Waiting loop - spin waiting for remote to complete something quick */
void    ipc_quickWait()
{
    int i = 0;
    i++;
}
/* Synchronize with remote processor
 * Only IPC_init should use this function */
int    ipc_InitSyncRemote (void)
{
    UInt16 volatile *ptr = (UInt16 volatile *) &ntfyCfgObj.remoteEventPtr->key;
    /* wait for remote system key to "appear" */
    while    (*ptr != IPC_BOOT_KEY)
        ipc_quickWait();
    return Notify_S_SUCCESS;
}

/* Synchronize with remote processor
 * Mechanism for local and remote CPUs to establish running state
 * This is used by Notify_start API to sync CPUs
 * Polling is used because it is the common denominator */
int     ipc_NotifySyncRemote (void)
{
    int val;
    UInt32 volatile *ptr = (UInt32 volatile *) &ntfyCfgObj.remoteEventPtr->payload;
    /* Put ready state in payload and wait here for remote ready */
    val = ntfyCfgObj.localEventPtr->payload = (UInt32) IPC_STATE_READY;
    while (val != *ptr)
        ipc_quickWait();
    return Notify_S_SUCCESS;
}

/* Notify send using SoC IPC interrupt */
int     ipc_NotifySendInterrupt(Bool waitClear)
{
    UInt16 volatile *ptr = (UInt16 volatile *) &ntfyCfgObj.remoteEventPtr->semCnt;
    int    status = Notify_S_SUCCESS;
    ipc_SetIpcInt();        /* generate interrupt */
    if    (waitClear    == true)
    {
        /* wait for remote CPU to finish ISR */
        while    (*ptr)
            ipc_quickWait();
    status = (int) (ntfyCfgObj.remoteEventPtr->payload & 0xFFFF);
    }
    return status;
}

/* Acknowledge an event notification from remote processor */
void    ipc_NotifyIsrAckEvent(void)
{
    register    int    st, evtId;

    evtId     = ntfyCfgObj.localEventPtr->eventId;
    st         = ntfyCfgObj.eventTblHandle[evtId].status;

    if    (st == Notify_EVENT_SLOT_REGISTED)
    {
        ntfyCfgObj.eventTblHandle[evtId].status = Notify_EVENT_SLOT_HAS_DATA;
        /* Call app's callback function */
        ntfyCfgObj.eventTblHandle[evtId].cbFxn(
                ntfyCfgObj.localEventPtr->procId,
                ntfyCfgObj.localEventPtr->lineId,
                evtId,
                ntfyCfgObj.eventTblHandle[evtId].cbKey,
                ntfyCfgObj.localEventPtr->payload        );
        /* Once returned, notify success status via payload */
        ntfyCfgObj.eventTblHandle[evtId].status = Notify_EVENT_SLOT_REGISTED;
        ntfyCfgObj.localEventPtr->payload = (UInt32) Notify_S_SUCCESS;
        ntfyCfgObj.pendingEventCnt++;
    }
    else if    (st == Notify_EVENT_SLOT_FREE)
    {
        ntfyCfgObj.localEventPtr->payload = (UInt32) Notify_E_EVTNOTREGISTERED;
    }
    else if    (st == Notify_EVENT_SLOT_DISABLED)
        ntfyCfgObj.localEventPtr->payload = (UInt32) Notify_E_EVTDISABLED;
    else if (st == Notify_EVENT_SLOT_HAS_DATA)
    {
        ntfyCfgObj.overrunEventCnt++;
        ntfyCfgObj.localEventPtr->payload = (UInt32) Notify_E_EVTOVERRUN;
    }
    /* last action - clear the semaphore */
    ntfyCfgObj.localEventPtr->semCnt = 0;
}

/* Save interrupt setup info
 * val = interrupt channel (ARM CPU AINTC) or interrupt number (C6x DSP INTC) */
void    ipc_NotifyIntInfo (int val)
{
    ntfyCfgObj.NotifyChannel = val;
}

/* Set up Notify control and data structures */
int ipc_NotifySetup(struct IPC_cfg *IpcObj, struct IPC_modControl *IpcCtl)
{
    int i;
    int    status = Notify_E_FAIL;

    /* Check maximum number of events (see define in Notify.h) */
    if     (IpcObj->maxEventNum <= Notify_MAXEVENTS)
    {
        /* Initialize private Notify control instance */
        ntfyCfgObj.NotifyRcvMethod    = IpcObj->ipcRcvMethod;
        ntfyCfgObj.maxNumofEvents     = IpcObj->maxEventNum;
        ntfyCfgObj.localEventPtr     = IpcObj->localEvtPtr;
        ntfyCfgObj.remoteEventPtr     = IpcObj->remoteEvtPtr;
        ntfyCfgObj.IpcObjHandle     = IpcCtl;
        ntfyCfgObj.numRegistedEvents = 0;
        ntfyCfgObj.pendingEventCnt     = 0;
        ntfyCfgObj.overrunEventCnt    = 0;
        IpcCtl->NotifyObjHandle = &ntfyCfgObj;

        /* Allocate space for local CPU event control structure */
/*    ntfyCfgObj.eventTblHandle = (IPC_notifyEventSlot *) malloc ( ntfyCfgObj.maxNumofEvents * sizeof(IPC_notifyEventSlot) ); */
/*  if (ntfyCfgObj.eventTblHandle == 0) */
/*        return status; */
        ntfyCfgObj.eventTblHandle = esTbl;

        /* Initialize the event slots for the local processor */
        for (i = 0; i <ntfyCfgObj.maxNumofEvents; i++)
        {
            ntfyCfgObj.eventTblHandle[i].status = Notify_EVENT_SLOT_FREE;
            ntfyCfgObj.eventTblHandle[i].remoteProcId = IpcObj->remoteProcId;
            ntfyCfgObj.eventTblHandle[i].remoteIntLine = IpcObj->remoteIntLine;
            ntfyCfgObj.eventTblHandle[i].semCnt = 0;
            ntfyCfgObj.eventTblHandle[i].cbKey = 0;
            ntfyCfgObj.eventTblHandle[i].cbFxn = EMPTY_PTR;
        }

        /* Select send driver */
        ntfyCfgObj.ipcNotifyFxn = ipc_NotifySendInterrupt;
        status = Notify_S_SUCCESS;
    }
    return (status);
}

/* END OF FILE */
