/**
 * @file ADS1158.c
 * @author James Thesing
 * @version 1.0
 *
 *@section LICENSE
 *
 * SPI testing utility (using spidev driver)
 *
 * Copyright (c) 2011  Critical Link LLC
 * 
 * This code is based on the software from the following:
 *
 * Copyright (c) 2007  MontaVista Software, Inc.
 * Copyright (c) 2007  Anton Vorontsov <avorontsov@ru.mvista.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License.
 *
 * Cross-compile with cross-gcc -I/path/to/cross-kernel/include
 *
 * @section DESCRIPTION
 *
 * C implementation that exercises the ADS1158 16-channel 16-bit ADC. 
 * Communication happens over the SPI interface. Results from the ADC
 * conversion are printed to standard output.
 */

#include <sys/types.h>
#include <stdint.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>


#define ARRAY_SIZE(a) (sizeof(a) / sizeof((a)[0]))



//spi defined variables
static const char *device = "/dev/spidev3.0";
static uint8_t mode = 0;
static uint8_t bits = 8;
static uint32_t numbytes = 5;
static uint32_t speed = 1000000;
static uint16_t delay = 0;
static uint8_t loopback = 0;
static uint32_t loops = 1;
static uint8_t quiet = 0;
static uint8_t tx[4];
static uint8_t rx[ARRAY_SIZE(tx)];

/**
 * Function that terminates operation when software has indicated a error
 *
 * @param *s Pointer to a c_string that contains the error msg
 */
static void pabort(const char *s)
{
	perror(s);
	abort();
}

/**
 * Transfers data over SPI
 *
 * @param fd SPI device identifier
 */
static int transfer(int fd)
{
	int ret, i, rv = 0;
	struct spi_ioc_transfer tr = {
		.tx_buf = (unsigned long)tx,
		.rx_buf = (unsigned long)rx,
		.len = numbytes,
		.delay_usecs = delay,
		.speed_hz = speed,
		.bits_per_word = bits,
	};

	ret = ioctl(fd, SPI_IOC_MESSAGE(1), &tr);
	if (ret == 1)
		pabort("can't send spi message");

	for (ret = 0; ret < numbytes; ret++) {
		if (loopback && (tx[ret] != rx[ret]))
		{
			rv = -1;
			break;
		}
	}
	
	if (rv)
		printf("ERROR DETECTED IN RECEIVE STREAM\n");

	return rv;
}

/**
 * Converts the 16-bit value from ADC into decimal notation
 *
 * @param c1 upper 8-bits of ADC output
 * @param c2 lower 8-bits of ADC output
 */
static double calcVoltage(uint8_t c1, uint8_t c0)
{
    //combine 8-bit values into 16-bit
	int16_t temp;
	double result;
	temp = (c1 << 8) | c0;
	//calculate decimal equivilent voltage
	if (temp > 0)
	{
		result = (double)temp / 7105 * 10;
	}
	else
	{
		result = (double)temp / 7579 * 10;
	}
	return result;
}

/**
 * Prints the usage of class to standard output
 *
 * @param prog pointer to the name of the program
 */
static void print_usage(const char *prog)
{
	printf("Usage: %s [-Dsdr] <CHAN#>\n", prog);
	puts("  -D --device   device to use (default /dev/spidev1.3)\n"
	     "  -s --speed    max speed (Hz default 5000000)\n"
	     "  -d --delay    delay (usec default = 0)\n"
	     "  -r --repeat   number of cycles to repeat(default = 0)\n"
		 "  CHAN# 1-16\n");
	exit(1);
}

/**
 * Parses user input, reports errors, and updates variables
 *
 * @param int argc the number of arguments
 * @param *argv[] the array of arguments
 */
static void parse_opts(int argc, char *argv[])
{
	while (1) {
		static const struct option lopts[] = {
			{ "device",  1, 0, 'D' },
			{ "speed",   1, 0, 's' },
			{ "delay",   1, 0, 'd' },
			{ "repeat",  1, 0, 'r' },
			{ NULL, 0, 0, 0 },
		};
		int c;

		c = getopt_long(argc, argv, "D:s:d:b:n:r:lqHOLC3NR", lopts, NULL);

		if (c == -1)
			break;

		switch (c) {
		case 'D':
			device = optarg;
			break;
		case 's':
			speed = atoi(optarg);
			break;
		case 'd':
			delay = atoi(optarg);
			break;
		case 'r':
			loops = atoi(optarg);
			if (loops <= 0) loops = 1;
			printf("running %d loops\n", loops);
			break;
		case 'n':
			numbytes = atoi(optarg);
			if (numbytes <= 0) numbytes = 38;
			if (numbytes > 512) numbytes = 512;
			break;
		default:
			print_usage(argv[0]);
			break;
		}
	}
}

/**
 * main function for running program
 *
 * @param int argc the number of arguments
 * @param *argv[] the array of arguments
 */
int main(int argc, char *argv[])
{
	int ret = 0;
	int fd, i, ch;
	char set_value[3];

	parse_opts(argc, argv);
	
	if (optind >= argc)
	{
		fprintf(stderr, "Not enough arguments\n");
		print_usage(argv[0]);
		return -1;
	}
	
	ch = atoi(argv[optind]);
	
	if ( (ch < 1) || (ch > 16) )
	{
		print_usage(argv[0]);
	}
	
	
	fd = open(device, O_RDWR);
	if (fd < 0)
		pabort("can't open device");
	/*
	 * spi mode
	 */
	ret = ioctl(fd, SPI_IOC_WR_MODE, &mode);
	if (ret == -1)
		pabort("can't set spi mode");

	ret = ioctl(fd, SPI_IOC_RD_MODE, &mode);
	if (ret == -1)
		pabort("can't get spi mode");

	/*
	 * bits per word
	 */
	ret = ioctl(fd, SPI_IOC_WR_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't set bits per word");

	ret = ioctl(fd, SPI_IOC_RD_BITS_PER_WORD, &bits);
	if (ret == -1)
		pabort("can't get bits per word");

	/*
	 * max speed hz
	 */
	ret = ioctl(fd, SPI_IOC_WR_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't set max speed hz");

	ret = ioctl(fd, SPI_IOC_RD_MAX_SPEED_HZ, &speed);
	if (ret == -1)
		pabort("can't get max speed hz");

	printf("spi mode: %d\n", mode);
	printf("bits per word: %d\n", bits);
	printf("max speed: %d Hz (%d KHz)\n", speed, speed/1000);
	
	if ( ret == -1 ){return ret;}
	
	uint8_t newStat = 0x80;
	uint8_t temp;
	ch = ch+7;
	while (1){
		tx[0] = 0x80;
		tx[1] = 0x30;
		ret = transfer(fd);
		temp = rx[2] & newStat;
		if( temp == newStat )
		{
			temp = rx[2] & 0x1F;
			if( temp == (ch) )
			{
				double volt = calcVoltage(rx[3], rx[4]);
				printf("the voltage of channel %d is %3.2f\n", ch - 7, volt);
				break;
			}
		}
	}

	close(fd);

	return ret;
}
