#include "tc74.h"
#include <stdio.h>
#include <stdint.h>
#include "i2c-dev.h"

using namespace MityDSP;

/* register definitions according to the TC74 data sheet */
/* http://ww1.microchip.com/downloads/en/DeviceDoc/21462c.pdf */

// bus info
static const int TC74_I2C_ADDR = 0x4D;

// registers
static const int TC74_TEMP   = 0;
static const int TC74_CONFIG = 1;

// config register map
static const int TC74_STANDBY = 7;
static const int TC74_READY = 6;


tcTC74::tcTC74(int bus)
: tcI2CDevice(bus, TC74_I2C_ADDR)
{
}

tcTC74::~tcTC74(void)
{
}

/**
 * Get the temperature from the device
 * The temp register is a _signed_ 8 bit value
 */
int tcTC74::getTemperature() const
{
    int8_t val;
    val = i2c_smbus_read_byte_data(mnFd, TC74_TEMP);
    return (int)val;
}

/**
 * get the standby state of the device
 * \return true if in standby
 */
bool tcTC74::getStandby() const
{
    int8_t val;
    val = i2c_smbus_read_byte_data(mnFd, TC74_CONFIG);
    return (bool)(val&(1 << TC74_STANDBY));
}

/**
 * put the device in standby mode or wake it up.
 * \param abStandby - true to go to standby mode
 */
int tcTC74::setStandby(bool abStandby) const
{
    // there is only 1 bit you can write in the config reg.
    return i2c_smbus_write_byte_data(mnFd,TC74_CONFIG,abStandby?(1 << TC74_STANDBY):0);
}


int tcTC74::Initialize(void)
{
	int ret;
	ret = Open();
    return ret;
}

#ifdef TC74_APP
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
int main(int argc, char* argv[])
{
    tcTC74* lpDev;
    int bus = 1;
    int opt;
    opterr = 0;
    
    while ((opt = getopt(argc, argv, "b:h")) != -1)
    {
        switch (opt)
        {
        case 'b':
            bus = strtoul(optarg,NULL,10);
            break;		

        case 'h': /* '?' */
        default: /* '?' */
            fprintf(stderr, "Usage: %s [-b busnum] <command>\n", argv[0]);
            fprintf(stderr, "Supported Commands\n"
                            "    temp        : read the temperature\n"
                            "    sleep [0|1] : put the device to sleep (1) or wake it up (0)\n"
                            );
            return -1;
        }
    }
    
    if (optind >= argc)
    {
        fprintf(stderr, "Not enough arguments\n");
    }
    
    lpDev = new tcTC74(bus);
    if (lpDev->Initialize())
    {
        fprintf(stderr, "unable to open device\n");
        delete lpDev;
        return -1;
    }
    
    if (!strncasecmp("temp", argv[optind],4))
    {
        int t = lpDev->getTemperature();
        fprintf(stdout, "Temperature = %d°C\n",t);
    }
    else if (!strncasecmp("sleep", argv[optind],5))
    {
        if (optind+1 < argc)
        {
            bool to_sleep = '0' == argv[optind+1][0];
            lpDev->setStandby(to_sleep);
        }
        else
        {
            bool to_sleep = lpDev->getStandby();
            fprintf(stdout,"TC74@%d:%x is%s in standby\n", lpDev->Bus(),lpDev->Addr(),to_sleep?"":"not");
        }
    }
    
    delete lpDev;
    return 0;
    
}
#endif
